"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildKinesisStreamCWAlarms = exports.buildKinesisStream = void 0;
// Imports
const kinesis = require("@aws-cdk/aws-kinesis");
const kinesis_streams_defaults_1 = require("./kinesis-streams-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
function buildKinesisStream(scope, props) {
    if (props.existingStreamObj) {
        return props.existingStreamObj;
    }
    // Setup the stream properties
    let kinesisStreamProps;
    if (props.kinesisStreamProps) {
        // If property overrides have been provided, incorporate them and deploy
        kinesisStreamProps = utils_1.overrideProps(kinesis_streams_defaults_1.DefaultStreamProps, props.kinesisStreamProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        kinesisStreamProps = kinesis_streams_defaults_1.DefaultStreamProps;
    }
    // Create the stream and return
    return new kinesis.Stream(scope, 'KinesisStream', kinesisStreamProps);
}
exports.buildKinesisStream = buildKinesisStream;
function buildKinesisStreamCWAlarms(scope) {
    // Setup CW Alarms for KinesisStream
    const alarms = new Array();
    // Alarm if Max (GetRecords.IteratorAgeMilliseconds): >= 12 hours (50% of 24 hours default retention period)
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamGetRecordsIteratorAgeAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'GetRecords.IteratorAgeMilliseconds',
            statistic: 'Maximum',
            period: cdk.Duration.minutes(5),
        }),
        threshold: 2592000,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Consumer Record Processing Falling Behind, there is risk for data loss due to record expiration.'
    }));
    // Alarm if Avg (ReadProvisionedThroughputExceeded): > 0
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamReadProvisionedThroughputExceededAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'ReadProvisionedThroughputExceeded',
            statistic: 'Average',
            period: cdk.Duration.minutes(5),
        }),
        threshold: 0,
        evaluationPeriods: 1,
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
        alarmDescription: 'Consumer Application is Reading at a Slower Rate Than Expected.'
    }));
    return alarms;
}
exports.buildKinesisStreamCWAlarms = buildKinesisStreamCWAlarms;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia2luZXNpcy1zdHJlYW1zLWhlbHBlci5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImtpbmVzaXMtc3RyZWFtcy1oZWxwZXIudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IjtBQUFBOzs7Ozs7Ozs7OztHQVdHOzs7QUFFSCxVQUFVO0FBQ1YsZ0RBQWdEO0FBQ2hELHlFQUFnRTtBQUNoRSxxQ0FBcUM7QUFDckMsbUNBQXdDO0FBQ3hDLHNEQUFzRDtBQVl0RCxTQUFnQixrQkFBa0IsQ0FBQyxLQUFnQixFQUFFLEtBQThCO0lBRWpGLElBQUksS0FBSyxDQUFDLGlCQUFpQixFQUFFO1FBQzNCLE9BQU8sS0FBSyxDQUFDLGlCQUFpQixDQUFDO0tBQ2hDO0lBRUQsOEJBQThCO0lBQzlCLElBQUksa0JBQWtCLENBQUM7SUFDdkIsSUFBSSxLQUFLLENBQUMsa0JBQWtCLEVBQUU7UUFDNUIsd0VBQXdFO1FBQ3hFLGtCQUFrQixHQUFHLHFCQUFhLENBQUMsNkNBQWtCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLENBQUM7S0FDbEY7U0FBTTtRQUNMLG1FQUFtRTtRQUNuRSxrQkFBa0IsR0FBRyw2Q0FBa0IsQ0FBQztLQUN6QztJQUVELCtCQUErQjtJQUMvQixPQUFPLElBQUksT0FBTyxDQUFDLE1BQU0sQ0FBQyxLQUFLLEVBQUUsZUFBZSxFQUFFLGtCQUFrQixDQUFDLENBQUM7QUFDeEUsQ0FBQztBQWxCRCxnREFrQkM7QUFFRCxTQUFnQiwwQkFBMEIsQ0FBQyxLQUFnQjtJQUN6RCxvQ0FBb0M7SUFDcEMsTUFBTSxNQUFNLEdBQXVCLElBQUksS0FBSyxFQUFFLENBQUM7SUFFL0MsNEdBQTRHO0lBQzVHLE1BQU0sQ0FBQyxJQUFJLENBQUMsSUFBSSxVQUFVLENBQUMsS0FBSyxDQUFDLEtBQUssRUFBRSx5Q0FBeUMsRUFBRTtRQUNqRixNQUFNLEVBQUUsSUFBSSxVQUFVLENBQUMsTUFBTSxDQUFDO1lBQzVCLFNBQVMsRUFBRSxhQUFhO1lBQ3hCLFVBQVUsRUFBRSxvQ0FBb0M7WUFDaEQsU0FBUyxFQUFFLFNBQVM7WUFDcEIsTUFBTSxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQztTQUNoQyxDQUFDO1FBQ0YsU0FBUyxFQUFFLE9BQU87UUFDbEIsaUJBQWlCLEVBQUUsQ0FBQztRQUNwQixrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsa0NBQWtDO1FBQ3BGLGdCQUFnQixFQUFFLGtHQUFrRztLQUNySCxDQUFDLENBQUMsQ0FBQztJQUVKLHdEQUF3RDtJQUN4RCxNQUFNLENBQUMsSUFBSSxDQUFDLElBQUksVUFBVSxDQUFDLEtBQUssQ0FBQyxLQUFLLEVBQUUscURBQXFELEVBQUU7UUFDN0YsTUFBTSxFQUFFLElBQUksVUFBVSxDQUFDLE1BQU0sQ0FBQztZQUM1QixTQUFTLEVBQUUsYUFBYTtZQUN4QixVQUFVLEVBQUUsbUNBQW1DO1lBQy9DLFNBQVMsRUFBRSxTQUFTO1lBQ3BCLE1BQU0sRUFBRSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7U0FDaEMsQ0FBQztRQUNGLFNBQVMsRUFBRSxDQUFDO1FBQ1osaUJBQWlCLEVBQUUsQ0FBQztRQUNwQixrQkFBa0IsRUFBRSxVQUFVLENBQUMsa0JBQWtCLENBQUMsc0JBQXNCO1FBQ3hFLGdCQUFnQixFQUFFLGlFQUFpRTtLQUNwRixDQUFDLENBQUMsQ0FBQztJQUVKLE9BQU8sTUFBTSxDQUFDO0FBQ2hCLENBQUM7QUFqQ0QsZ0VBaUNDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiAgQ29weXJpZ2h0IDIwMjEgQW1hem9uLmNvbSwgSW5jLiBvciBpdHMgYWZmaWxpYXRlcy4gQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS4gWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZVxuICogIHdpdGggdGhlIExpY2Vuc2UuIEEgY29weSBvZiB0aGUgTGljZW5zZSBpcyBsb2NhdGVkIGF0XG4gKlxuICogICAgICBodHRwOi8vd3d3LmFwYWNoZS5vcmcvbGljZW5zZXMvTElDRU5TRS0yLjBcbiAqXG4gKiAgb3IgaW4gdGhlICdsaWNlbnNlJyBmaWxlIGFjY29tcGFueWluZyB0aGlzIGZpbGUuIFRoaXMgZmlsZSBpcyBkaXN0cmlidXRlZCBvbiBhbiAnQVMgSVMnIEJBU0lTLCBXSVRIT1VUIFdBUlJBTlRJRVNcbiAqICBPUiBDT05ESVRJT05TIE9GIEFOWSBLSU5ELCBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZyBwZXJtaXNzaW9uc1xuICogIGFuZCBsaW1pdGF0aW9ucyB1bmRlciB0aGUgTGljZW5zZS5cbiAqL1xuXG4vLyBJbXBvcnRzXG5pbXBvcnQgKiBhcyBraW5lc2lzIGZyb20gJ0Bhd3MtY2RrL2F3cy1raW5lc2lzJztcbmltcG9ydCB7IERlZmF1bHRTdHJlYW1Qcm9wcyB9IGZyb20gJy4va2luZXNpcy1zdHJlYW1zLWRlZmF1bHRzJztcbmltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IG92ZXJyaWRlUHJvcHMgfSBmcm9tICcuL3V0aWxzJztcbmltcG9ydCAqIGFzIGNsb3Vkd2F0Y2ggZnJvbSAnQGF3cy1jZGsvYXdzLWNsb3Vkd2F0Y2gnO1xuLy8gTm90ZTogVG8gZW5zdXJlIENES3YyIGNvbXBhdGliaWxpdHksIGtlZXAgdGhlIGltcG9ydCBzdGF0ZW1lbnQgZm9yIENvbnN0cnVjdCBzZXBhcmF0ZVxuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSAnQGF3cy1jZGsvY29yZSc7XG5cbmV4cG9ydCBpbnRlcmZhY2UgQnVpbGRLaW5lc2lzU3RyZWFtUHJvcHMge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICAgIHJlYWRvbmx5IGV4aXN0aW5nU3RyZWFtT2JqPzoga2luZXNpcy5TdHJlYW07XG5cbiAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIFxuICByZWFkb25seSBraW5lc2lzU3RyZWFtUHJvcHM/OiBraW5lc2lzLlN0cmVhbVByb3BzXG59XG5cbmV4cG9ydCBmdW5jdGlvbiBidWlsZEtpbmVzaXNTdHJlYW0oc2NvcGU6IENvbnN0cnVjdCwgcHJvcHM6IEJ1aWxkS2luZXNpc1N0cmVhbVByb3BzKToga2luZXNpcy5TdHJlYW0ge1xuXG4gIGlmIChwcm9wcy5leGlzdGluZ1N0cmVhbU9iaikge1xuICAgIHJldHVybiBwcm9wcy5leGlzdGluZ1N0cmVhbU9iajtcbiAgfVxuXG4gIC8vIFNldHVwIHRoZSBzdHJlYW0gcHJvcGVydGllc1xuICBsZXQga2luZXNpc1N0cmVhbVByb3BzO1xuICBpZiAocHJvcHMua2luZXNpc1N0cmVhbVByb3BzKSB7XG4gICAgLy8gSWYgcHJvcGVydHkgb3ZlcnJpZGVzIGhhdmUgYmVlbiBwcm92aWRlZCwgaW5jb3Jwb3JhdGUgdGhlbSBhbmQgZGVwbG95XG4gICAga2luZXNpc1N0cmVhbVByb3BzID0gb3ZlcnJpZGVQcm9wcyhEZWZhdWx0U3RyZWFtUHJvcHMsIHByb3BzLmtpbmVzaXNTdHJlYW1Qcm9wcyk7XG4gIH0gZWxzZSB7XG4gICAgLy8gSWYgbm8gcHJvcGVydHkgb3ZlcnJpZGVzLCBkZXBsb3kgdXNpbmcgdGhlIGRlZmF1bHQgY29uZmlndXJhdGlvblxuICAgIGtpbmVzaXNTdHJlYW1Qcm9wcyA9IERlZmF1bHRTdHJlYW1Qcm9wcztcbiAgfVxuXG4gIC8vIENyZWF0ZSB0aGUgc3RyZWFtIGFuZCByZXR1cm5cbiAgcmV0dXJuIG5ldyBraW5lc2lzLlN0cmVhbShzY29wZSwgJ0tpbmVzaXNTdHJlYW0nLCBraW5lc2lzU3RyZWFtUHJvcHMpO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gYnVpbGRLaW5lc2lzU3RyZWFtQ1dBbGFybXMoc2NvcGU6IENvbnN0cnVjdCk6IGNsb3Vkd2F0Y2guQWxhcm1bXSB7XG4gIC8vIFNldHVwIENXIEFsYXJtcyBmb3IgS2luZXNpc1N0cmVhbVxuICBjb25zdCBhbGFybXM6IGNsb3Vkd2F0Y2guQWxhcm1bXSA9IG5ldyBBcnJheSgpO1xuXG4gIC8vIEFsYXJtIGlmIE1heCAoR2V0UmVjb3Jkcy5JdGVyYXRvckFnZU1pbGxpc2Vjb25kcyk6ID49IDEyIGhvdXJzICg1MCUgb2YgMjQgaG91cnMgZGVmYXVsdCByZXRlbnRpb24gcGVyaW9kKVxuICBhbGFybXMucHVzaChuZXcgY2xvdWR3YXRjaC5BbGFybShzY29wZSwgJ0tpbmVzaXNTdHJlYW1HZXRSZWNvcmRzSXRlcmF0b3JBZ2VBbGFybScsIHtcbiAgICBtZXRyaWM6IG5ldyBjbG91ZHdhdGNoLk1ldHJpYyh7XG4gICAgICBuYW1lc3BhY2U6ICdBV1MvS2luZXNpcycsXG4gICAgICBtZXRyaWNOYW1lOiAnR2V0UmVjb3Jkcy5JdGVyYXRvckFnZU1pbGxpc2Vjb25kcycsXG4gICAgICBzdGF0aXN0aWM6ICdNYXhpbXVtJyxcbiAgICAgIHBlcmlvZDogY2RrLkR1cmF0aW9uLm1pbnV0ZXMoNSksXG4gICAgfSksXG4gICAgdGhyZXNob2xkOiAyNTkyMDAwLCAvLyAxMiBIb3VycyAoNTAlIG9mIDI0IGhvdXJzIC0gZGVmYXVsdCByZWNvcmQgcmV0ZW50aW9uIHBlcmlvZClcbiAgICBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICBjb21wYXJpc29uT3BlcmF0b3I6IGNsb3Vkd2F0Y2guQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9PUl9FUVVBTF9UT19USFJFU0hPTEQsXG4gICAgYWxhcm1EZXNjcmlwdGlvbjogJ0NvbnN1bWVyIFJlY29yZCBQcm9jZXNzaW5nIEZhbGxpbmcgQmVoaW5kLCB0aGVyZSBpcyByaXNrIGZvciBkYXRhIGxvc3MgZHVlIHRvIHJlY29yZCBleHBpcmF0aW9uLidcbiAgfSkpO1xuXG4gIC8vIEFsYXJtIGlmIEF2ZyAoUmVhZFByb3Zpc2lvbmVkVGhyb3VnaHB1dEV4Y2VlZGVkKTogPiAwXG4gIGFsYXJtcy5wdXNoKG5ldyBjbG91ZHdhdGNoLkFsYXJtKHNjb3BlLCAnS2luZXNpc1N0cmVhbVJlYWRQcm92aXNpb25lZFRocm91Z2hwdXRFeGNlZWRlZEFsYXJtJywge1xuICAgIG1ldHJpYzogbmV3IGNsb3Vkd2F0Y2guTWV0cmljKHtcbiAgICAgIG5hbWVzcGFjZTogJ0FXUy9LaW5lc2lzJyxcbiAgICAgIG1ldHJpY05hbWU6ICdSZWFkUHJvdmlzaW9uZWRUaHJvdWdocHV0RXhjZWVkZWQnLFxuICAgICAgc3RhdGlzdGljOiAnQXZlcmFnZScsXG4gICAgICBwZXJpb2Q6IGNkay5EdXJhdGlvbi5taW51dGVzKDUpLFxuICAgIH0pLFxuICAgIHRocmVzaG9sZDogMCxcbiAgICBldmFsdWF0aW9uUGVyaW9kczogMSxcbiAgICBjb21wYXJpc29uT3BlcmF0b3I6IGNsb3Vkd2F0Y2guQ29tcGFyaXNvbk9wZXJhdG9yLkdSRUFURVJfVEhBTl9USFJFU0hPTEQsXG4gICAgYWxhcm1EZXNjcmlwdGlvbjogJ0NvbnN1bWVyIEFwcGxpY2F0aW9uIGlzIFJlYWRpbmcgYXQgYSBTbG93ZXIgUmF0ZSBUaGFuIEV4cGVjdGVkLidcbiAgfSkpO1xuXG4gIHJldHVybiBhbGFybXM7XG59Il19