"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.createGlueDatabase = exports.createGlueTable = exports.createGlueJobRole = exports.deployGlueJob = exports.buildGlueJob = exports.SinkStoreType = void 0;
const glue = require("@aws-cdk/aws-glue");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_1 = require("@aws-cdk/aws-s3");
const core_1 = require("@aws-cdk/core");
const defaults = require("../");
const utils_1 = require("./utils");
/**
 * Enumeration of data store types that could include S3, DynamoDB, DocumentDB, RDS or Redshift. Current
 * construct implementation only supports S3, but potential to add other output types in the future
 */
var SinkStoreType;
(function (SinkStoreType) {
    SinkStoreType["S3"] = "S3";
})(SinkStoreType = exports.SinkStoreType || (exports.SinkStoreType = {}));
function buildGlueJob(scope, props) {
    if (!props.existingCfnJob) {
        if (props.glueJobProps) {
            if (props.glueJobProps.glueVersion === '2.0' && props.glueJobProps.maxCapacity) {
                throw Error('Cannot set "MaxCapacity" with GlueVersion 2.0 or higher. Use "NumberOfWorkers" and "WorkerType". ' +
                    'Refer the API documentation https://docs.aws.amazon.com/glue/latest/webapi/API_Job.html for more details');
            }
            if (props.glueJobProps.maxCapacity && (props.glueJobProps.numberOfWorkers || props.glueJobProps.workerType)) {
                throw Error('Cannot set MaxCapacity and "WorkerType" or  "NumberOfWorkers". If using glueVersion 2.0 or beyond, ' +
                    'it is recommended to use "WorkerType" or  "NumberOfWorkers"');
            }
            return deployGlueJob(scope, props.glueJobProps, props.database, props.table, props.outputDataStore);
        }
        else {
            throw Error('Either glueJobProps or existingCfnJob is required');
        }
    }
    else {
        return [props.existingCfnJob, aws_iam_1.Role.fromRoleArn(scope, 'ExistingRole', props.existingCfnJob.role)];
    }
}
exports.buildGlueJob = buildGlueJob;
function deployGlueJob(scope, glueJobProps, database, table, outputDataStore) {
    let _glueSecurityConfigName;
    if (glueJobProps.securityConfiguration === undefined) {
        _glueSecurityConfigName = 'ETLJobSecurityConfig';
        const _glueKMSKey = `arn:${core_1.Aws.PARTITION}:kms:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:alias/aws/glue`;
        new glue.CfnSecurityConfiguration(scope, 'GlueSecurityConfig', {
            name: _glueSecurityConfigName,
            encryptionConfiguration: {
                jobBookmarksEncryption: {
                    jobBookmarksEncryptionMode: 'CSE-KMS',
                    kmsKeyArn: _glueKMSKey
                },
                s3Encryptions: [{
                        s3EncryptionMode: 'SSE-S3'
                    }]
            }
        });
    }
    else {
        _glueSecurityConfigName = glueJobProps.securityConfiguration;
    }
    const _glueJobPolicy = new aws_iam_1.Policy(scope, 'LogPolicy', {
        statements: [
            new aws_iam_1.PolicyStatement({
                effect: aws_iam_1.Effect.ALLOW,
                actions: ['logs:CreateLogGroup', 'logs:CreateLogStream', 'logs:PutLogEvents'],
                resources: [`arn:${core_1.Aws.PARTITION}:logs:${core_1.Aws.REGION}:${core_1.Aws.ACCOUNT_ID}:log-group:/aws-glue/*`]
            })
        ]
    });
    let _jobRole;
    if (glueJobProps.role) {
        _jobRole = aws_iam_1.Role.fromRoleArn(scope, 'JobRole', glueJobProps.role);
    }
    else {
        _jobRole = defaults.createGlueJobRole(scope);
    }
    _glueJobPolicy.attachToRole(_jobRole);
    let _outputLocation;
    if (outputDataStore !== undefined && outputDataStore.datastoreType === SinkStoreType.S3) {
        if (outputDataStore.existingS3OutputBucket !== undefined) {
            _outputLocation = [outputDataStore.existingS3OutputBucket, undefined];
        }
        else {
            _outputLocation = defaults.buildS3Bucket(scope, { bucketProps: outputDataStore.outputBucketProps });
        }
    }
    else {
        _outputLocation = defaults.buildS3Bucket(scope, {});
    }
    _outputLocation[0].grantReadWrite(_jobRole);
    const _jobArgumentsList = {
        "--enable-metrics": true,
        "--enable-continuous-cloudwatch-log": true,
        "--database_name": database.ref,
        "--table_name": table.ref,
        ...((outputDataStore === undefined || (outputDataStore && outputDataStore.datastoreType === SinkStoreType.S3)) &&
            { '--output_path': `s3a://${_outputLocation[0].bucketName}/output/` }),
        ...glueJobProps.defaultArguments
    };
    const _newGlueJobProps = utils_1.overrideProps(defaults.DefaultGlueJobProps(_jobRole, glueJobProps, _glueSecurityConfigName, _jobArgumentsList), glueJobProps);
    let _scriptLocation;
    if (isJobCommandProperty(_newGlueJobProps.command)) {
        if (_newGlueJobProps.command.scriptLocation) {
            _scriptLocation = _newGlueJobProps.command.scriptLocation;
        }
        else {
            throw Error('Script location has to be provided as an s3 Url location. Script location cannot be empty');
        }
    }
    const _scriptBucketLocation = aws_s3_1.Bucket.fromBucketArn(scope, 'ScriptLocaiton', getS3ArnfromS3Url(_scriptLocation));
    _scriptBucketLocation.grantRead(_jobRole);
    const _glueJob = new glue.CfnJob(scope, 'KinesisETLJob', _newGlueJobProps);
    return [_glueJob, _jobRole];
}
exports.deployGlueJob = deployGlueJob;
/**
 * This is a helper method to create the Role required for the Glue Job. If a role is already created then this
 * method is not required to be called.
 *
 * @param scope - The AWS Construct under which the role is to be created
 */
function createGlueJobRole(scope) {
    return new aws_iam_1.Role(scope, 'JobRole', {
        assumedBy: new aws_iam_1.ServicePrincipal('glue.amazonaws.com'),
        description: 'Service role that Glue custom ETL jobs will assume for exeuction',
    });
}
exports.createGlueJobRole = createGlueJobRole;
/**
 * This method creates an AWS Glue table. The method is called when an existing Glue table is not provided
 */
function createGlueTable(scope, database, tableProps, fieldSchema, sourceType, parameters) {
    return defaults.DefaultGlueTable(scope, tableProps !== undefined ? tableProps :
        defaults.DefaultGlueTableProps(database, fieldSchema, sourceType, parameters));
}
exports.createGlueTable = createGlueTable;
/**
 * This method creates an AWS Glue database. The method is only called with an existing Glue database type is not provided.
 * The method uses the user provided props to override the defaul props for the Glue database
 *
 * @param scope
 * @param databaseProps
 */
function createGlueDatabase(scope, databaseProps) {
    const _mergedDBProps = (databaseProps !== undefined) ? utils_1.overrideProps(defaults.DefaultGlueDatabaseProps(), databaseProps) :
        defaults.DefaultGlueDatabaseProps();
    return defaults.DefaultGlueDatabase(scope, _mergedDBProps);
}
exports.createGlueDatabase = createGlueDatabase;
/**
 * A utility method to generate the S3 Arn from an S3 Url.
 *
 * @param s3Url
 */
function getS3ArnfromS3Url(s3Url) {
    const splitString = s3Url.slice('s3://'.length);
    return `arn:${core_1.Aws.PARTITION}:s3:::${splitString}`;
}
/**
 * A utility method to type check CfnJob.JobCommandProperty type.
 *
 * @param command
 */
function isJobCommandProperty(command) {
    if (command.name ||
        command.pythonVersion ||
        command.scriptLocation) {
        return true;
    }
    else {
        defaults.printWarning('command not of type JobCommandProperty type');
        return false;
    }
}
//# sourceMappingURL=data:application/json;base64,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