"use strict";
/**
 *  Copyright 2021 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.addProxyMethodToApiResource = exports.RegionalRestApi = exports.GlobalRestApi = exports.RegionalLambdaRestApi = exports.GlobalLambdaRestApi = void 0;
const cdk = require("@aws-cdk/core");
const api = require("@aws-cdk/aws-apigateway");
const iam = require("@aws-cdk/aws-iam");
const apiDefaults = require("./apigateway-defaults");
const cloudwatch_log_group_helper_1 = require("./cloudwatch-log-group-helper");
const utils_1 = require("./utils");
/**
 * Create and configures access logging for API Gateway resources.
 * @param scope - the construct to which the access logging capabilities should be attached to.
 * @param _api - an existing api.RestApi or api.LambdaRestApi.
 */
function configureCloudwatchRoleForApi(scope, _api) {
    var _a;
    // Setup the IAM Role for API Gateway CloudWatch access
    const restApiCloudwatchRole = new iam.Role(scope, 'LambdaRestApiCloudWatchRole', {
        assumedBy: new iam.ServicePrincipal('apigateway.amazonaws.com'),
        inlinePolicies: {
            LambdaRestApiCloudWatchRolePolicy: new iam.PolicyDocument({
                statements: [new iam.PolicyStatement({
                        actions: [
                            'logs:CreateLogGroup',
                            'logs:CreateLogStream',
                            'logs:DescribeLogGroups',
                            'logs:DescribeLogStreams',
                            'logs:PutLogEvents',
                            'logs:GetLogEvents',
                            'logs:FilterLogEvents'
                        ],
                        resources: [`arn:${cdk.Aws.PARTITION}:logs:${cdk.Aws.REGION}:${cdk.Aws.ACCOUNT_ID}:*`]
                    })]
            })
        }
    });
    // Create and configure AWS::ApiGateway::Account with CloudWatch Role for ApiGateway
    const CfnApi = _api.node.findChild('Resource');
    const cfnAccount = new api.CfnAccount(scope, 'LambdaRestApiAccount', {
        cloudWatchRoleArn: restApiCloudwatchRole.roleArn
    });
    cfnAccount.addDependsOn(CfnApi);
    // Suppress Cfn Nag warning for APIG
    const deployment = (_a = _api.latestDeployment) === null || _a === void 0 ? void 0 : _a.node.findChild('Resource');
    utils_1.addCfnSuppressRules(deployment, [
        {
            id: 'W45',
            reason: `ApiGateway has AccessLogging enabled in AWS::ApiGateway::Stage resource, but cfn_nag checkes for it in AWS::ApiGateway::Deployment resource`
        }
    ]);
    // Return the CW Role
    return restApiCloudwatchRole;
}
/**
 * Creates and configures an api.LambdaRestApi.
 * @param scope - the construct to which the LambdaRestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the LambdaRestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureLambdaRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API object
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, apiGatewayProps);
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.LambdaRestApi(scope, 'LambdaRestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    const cwRole = configureCloudwatchRoleForApi(scope, _api);
    let usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        const extraParams = { apiKey: _api.addApiKey('ApiKey') };
        usagePlanProps = Object.assign(usagePlanProps, extraParams);
    }
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', usagePlanProps);
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Creates and configures an api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param defaultApiGatewayProps - the default properties for the RestApi.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function configureRestApi(scope, defaultApiGatewayProps, apiGatewayProps) {
    var _a;
    // API Gateway doesn't allow both endpointTypes and endpointConfiguration, check whether endPointTypes exists
    if (apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.endpointTypes) {
        throw Error('Solutions Constructs internally uses endpointConfiguration, use endpointConfiguration instead of endpointTypes');
    }
    // Define the API
    let _api;
    if (apiGatewayProps) {
        // If property overrides have been provided, incorporate them and deploy
        const _apiGatewayProps = utils_1.overrideProps(defaultApiGatewayProps, apiGatewayProps);
        _api = new api.RestApi(scope, 'RestApi', _apiGatewayProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        _api = new api.RestApi(scope, 'RestApi', defaultApiGatewayProps);
    }
    // Configure API access logging
    const cwRole = configureCloudwatchRoleForApi(scope, _api);
    let usagePlanProps = {
        apiStages: [{
                api: _api,
                stage: _api.deploymentStage
            }]
    };
    // If requireApiKey param is set to true, create a api key & associate to Usage Plan
    if (((_a = apiGatewayProps === null || apiGatewayProps === void 0 ? void 0 : apiGatewayProps.defaultMethodOptions) === null || _a === void 0 ? void 0 : _a.apiKeyRequired) === true) {
        const extraParams = { apiKey: _api.addApiKey('ApiKey') };
        usagePlanProps = Object.assign(usagePlanProps, extraParams);
    }
    // Configure Usage Plan
    _api.addUsagePlan('UsagePlan', usagePlanProps);
    // Return the API and CW Role
    return [_api, cwRole];
}
/**
 * Builds and returns a global api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalLambdaRestApi = GlobalLambdaRestApi;
/**
 * Builds and returns a regional api.RestApi designed to be used with an AWS Lambda function.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param _existingLambdaObj - an existing AWS Lambda function.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalLambdaRestApi(scope, _existingLambdaObj, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalLambdaRestApiProps(_existingLambdaObj, logGroup);
    const [restApi, apiCWRole] = configureLambdaRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalLambdaRestApi = RegionalLambdaRestApi;
/**
 * Builds and returns a standard api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function GlobalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultGlobalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.GlobalRestApi = GlobalRestApi;
/**
 * Builds and returns a Regional api.RestApi.
 * @param scope - the construct to which the RestApi should be attached to.
 * @param apiGatewayProps - (optional) user-specified properties to override the default properties.
 */
function RegionalRestApi(scope, apiGatewayProps, logGroupProps) {
    // Configure log group for API Gateway AccessLogging
    const logGroup = cloudwatch_log_group_helper_1.buildLogGroup(scope, 'ApiAccessLogGroup', logGroupProps);
    const defaultProps = apiDefaults.DefaultRegionalRestApiProps(logGroup);
    const [restApi, apiCWRole] = configureRestApi(scope, defaultProps, apiGatewayProps);
    return [restApi, apiCWRole, logGroup];
}
exports.RegionalRestApi = RegionalRestApi;
function addProxyMethodToApiResource(params) {
    let baseProps = {
        service: params.service,
        integrationHttpMethod: "POST",
        options: {
            passthroughBehavior: api.PassthroughBehavior.NEVER,
            credentialsRole: params.apiGatewayRole,
            requestParameters: {
                "integration.request.header.Content-Type": params.contentType ? params.contentType : "'application/json'"
            },
            requestTemplates: {
                "application/json": params.requestTemplate
            },
            integrationResponses: [
                {
                    statusCode: "200"
                },
                {
                    statusCode: "500",
                    responseTemplates: {
                        "text/html": "Error"
                    },
                    selectionPattern: "500"
                }
            ]
        }
    };
    let extraProps;
    if (params.action) {
        extraProps = {
            action: params.action
        };
    }
    else if (params.path) {
        extraProps = {
            path: params.path
        };
    }
    else {
        throw Error('Either action or path is required');
    }
    // Setup the API Gateway AWS Integration
    baseProps = Object.assign(baseProps, extraProps);
    let apiGatewayIntegration;
    if (params.awsIntegrationProps) {
        const overridenProps = utils_1.overrideProps(baseProps, params.awsIntegrationProps);
        apiGatewayIntegration = new api.AwsIntegration(overridenProps);
    }
    else {
        apiGatewayIntegration = new api.AwsIntegration(baseProps);
    }
    const defaultMethodOptions = {
        methodResponses: [
            {
                statusCode: "200",
                responseParameters: {
                    "method.response.header.Content-Type": true
                }
            },
            {
                statusCode: "500",
                responseParameters: {
                    "method.response.header.Content-Type": true
                },
            }
        ],
        requestValidator: params.requestValidator,
        requestModels: params.requestModel
    };
    let apiMethod;
    // Setup the API Gateway method
    if (params.methodOptions) {
        const overridenProps = utils_1.overrideProps(defaultMethodOptions, params.methodOptions);
        apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, overridenProps);
    }
    else {
        apiMethod = params.apiResource.addMethod(params.apiMethod, apiGatewayIntegration, defaultMethodOptions);
    }
    return apiMethod;
}
exports.addProxyMethodToApiResource = addProxyMethodToApiResource;
//# sourceMappingURL=data:application/json;base64,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