# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['librelingo_types']

package_data = \
{'': ['*']}

setup_kwargs = {
    'name': 'librelingo-types',
    'version': '3.3.0',
    'description': 'Data types to be used in Python packages for LibreLingo',
    'long_description': '<a name="librelingo_types"></a>\n# librelingo\\_types\n\nData types to be used in Python packages for LibreLingo\n\n<a name="librelingo_types.data_types"></a>\n# librelingo\\_types.data\\_types\n\n<a name="librelingo_types.data_types.TextToSpeechSettings"></a>\n## TextToSpeechSettings Objects\n\n```python\nclass TextToSpeechSettings(\n    namedtuple(\n        "TextToSpeechSettings",\n        [\n            "provider",\n            "voice",\n            "engine",\n        ],\n        defaults=["Polly", "Lupe", "standard"],\n    ))\n```\n\nSettings about how to use TTS to generate audios\n\n### Usage example:\n\n>>> TextToSpeechSettings(\n...     provider="Polly",\n...     voice="Aditi",\n...     engine="standard"\n... )\nTextToSpeechSettings(provider=\'Polly\', voice=\'Aditi\', engine=\'standard\')\n\n<a name="librelingo_types.data_types.AudioSettings"></a>\n## AudioSettings Objects\n\n```python\nclass AudioSettings(\n    namedtuple(\n        "AudioSettings",\n        [\n            "enabled",\n            "text_to_speech_settings_list",\n        ],\n        defaults=[False, []],\n    ))\n```\n\nSettings for audio in a course\n\n### Usage example:\n\n>>> AudioSettings(\n...     enabled=True,\n...     text_to_speech_settings_list=[TextToSpeechSettings()]\n... )\nAudioSettings(enabled=True, text_to_speech_settings_list=[TextToSpeechSettings(provider=\'Polly\', voice=\'Lupe\', engine=\'standard\')])\n\n<a name="librelingo_types.data_types.HunspellSettings"></a>\n## HunspellSettings Objects\n\n```python\nclass HunspellSettings(\n    namedtuple(\n        "HunspellSettings",\n        [\n            "source_language",\n            "target_language",\n        ],\n        defaults=[None, None],\n    ))\n```\n\nSettings for hunspell spell checking\n\n### Usage example:\n\n>>> HunspellSettings(\n...     source_language="en-US",\n...     target_language="es-ES",\n... )\nHunspellSettings(source_language=\'en-US\', target_language=\'es-ES\')\n\n<a name="librelingo_types.data_types.Settings"></a>\n## Settings Objects\n\n```python\nclass Settings(\n    namedtuple(\n        "Settings",\n        ["audio_settings", "hunspell"],\n        defaults=[AudioSettings(), HunspellSettings()],\n    ))\n```\n\nSettings for a course\n\n### Usage example:\n\n>>> Settings()\nSettings(audio_settings=AudioSettings(enabled=False, text_to_speech_settings_list=[]), hunspell=HunspellSettings(source_language=None, target_language=None))\n\n<a name="librelingo_types.data_types.Course"></a>\n## Course Objects\n\n```python\nclass Course(\n    namedtuple(\n        "Course",\n        [\n            "target_language",\n            "source_language",\n            "special_characters",\n            "modules",\n            "license",\n            "dictionary",\n            "repository_url",\n            "course_dir",\n            "settings",\n        ],\n        defaults=[Settings()],\n    ))\n```\n\nA LibreLingo course\n\n### Usage example:\n\n```python\nmy_course = Course(\n    target_language=Language("English", "en"),\n    source_language=Language("Spanish", "es"),\n    special_characters=[],\n    modules=[module1, module2, module3, module4],\n    license=License(\n        full_name="Attribution 4.0 International (CC BY 4.0)",\n        name="CC BY 4.0",\n        link="https://creativecommons.org/licenses/by/4.0/"\n    ),\n    dictionary=[dict_item1, dict_item2, dict_item3, dict_item4],\n    repository_url="https://example.com",\n    course_dir="some_language/course",\n    settings=Settings()\n)\n```\n\n<a name="librelingo_types.data_types.Language"></a>\n## Language Objects\n\n```python\nclass Language(namedtuple("Language", ["name", "code"]))\n```\n\nMetadata about a language\n\n### Usage example:\n\n>>> Language("English", "en")\nLanguage(name=\'English\', code=\'en\')\n\n<a name="librelingo_types.data_types.License"></a>\n## License Objects\n\n```python\nclass License(\n    namedtuple(\n        "License",\n        [\n            "name",\n            "full_name",\n            "link",\n        ],\n    ))\n```\n\nMetadata about the license of a LibreLingo course\n\n### Usage example:\n\n>>> License(\n...     full_name="Attribution 4.0 International (CC BY 4.0)",\n...     name="CC BY 4.0",\n...     link="https://creativecommons.org/licenses/by/4.0/"\n... )\nLicense(name=\'CC BY 4.0\', full_name=\'Attribution 4.0 International (CC BY 4.0)\', link=\'https://creativecommons.org/licenses/by/4.0/\')\n\n<a name="librelingo_types.data_types.Module"></a>\n## Module Objects\n\n```python\nclass Module(\n    namedtuple(\n        "Module",\n        [\n            "title",\n            "filename",\n            "skills",\n        ],\n    ))\n```\n\nA module of a LibreLingo course.\n\n### Usage examples:\n\n```python\nmy_module = Module(title="Basics", filename="basic/module.yaml", skills=[skill1, skill2])\n```\n\n<a name="librelingo_types.data_types.Skill"></a>\n## Skill Objects\n\n```python\nclass Skill(\n    namedtuple(\n        "Skill",\n        [\n            "name",\n            "filename",\n            "id",\n            "words",\n            "phrases",\n            "image_set",\n            "dictionary",\n            "introduction",\n        ],\n    ))\n```\n\nA skill of a module of a LibreLingo course.\n\n### Notes:\n\n*id*: Needs to be a unique ID. Use uuidv4.\n\n### Usage examples:\n\n```python\nmy_skill = Skill(\n    name="Animals",\n    filename="basic/skills/hello.yaml",\n    id="3adc78da-ea42-4ecd-9e3d-2e0986a3b914",\n    words=[word1, word2, word3],\n    phrases=[phrases1, phrases2, phrases3],\n    image_set=["cat1", "dog2", "horse1"],\n    dictionary=[dict_item_1, dict_item_2, dict_item_3, dict_item_4],\n    introduction="My *markdown* text",\n)\n```\n\n<a name="librelingo_types.data_types.Word"></a>\n## Word Objects\n\n```python\nclass Word(\n    namedtuple(\n        "Word",\n        [\n            "in_target_language",\n            "in_source_language",\n            "pictures",\n        ],\n    ))\n```\n\nA new word taught in a LibreLingo skill.\n\n### Notes:\n\n*in_source_language*: List of accepted forms in the target language of the\ncourse. The first item in the list is the main form. The main form is the\nonly form that is shown in the course but all forms are accepted as\nanswers.\n\n*in_target_language*: List of accepted forms in the target language of the\ncourse. The first item in the list is the main form. The main form is the\nonly form that is shown in the course but all forms are accepted as\nanswers.\n\n### Usage example:\n\n>>> Word(\n...     in_target_language=["perro"],\n...     in_source_language=["dog"],\n...     pictures=["dog1", "dog2", "dog3"]\n... )\nWord(in_target_language=[\'perro\'], in_source_language=[\'dog\'], pictures=[\'dog1\', \'dog2\', \'dog3\'])\n\n<a name="librelingo_types.data_types.Phrase"></a>\n## Phrase Objects\n\n```python\nclass Phrase(\n    namedtuple(\n        "Phrase",\n        [\n            "in_target_language",\n            "in_source_language",\n        ],\n    ))\n```\n\nA new phrase taught in a LibreLingo skill.\n\n### Notes:\n\n*in_source_language*: List of accepted forms in the target language of the\ncourse. The first item in the list is the main form. The main form is the\nonly form that is shown in the course but all forms are accepted as\nanswers.\n\n*in_target_language*: List of accepted forms in the target language of the\ncourse. The first item in the list is the main form. The main form is the\nonly form that is shown in the course but all forms are accepted as\nanswers.\n\n### Usage example:\n\n>>> Phrase(\n...     in_target_language=["perro", "can"],\n...     in_source_language=["dog"],\n... )\nPhrase(in_target_language=[\'perro\', \'can\'], in_source_language=[\'dog\'])\n\n<a name="librelingo_types.data_types.DictionaryItem"></a>\n## DictionaryItem Objects\n\n```python\nclass DictionaryItem(\n    namedtuple("DictionaryItem", ["word", "definition", "is_in_target_language"]))\n```\n\nA dictionary item for a LibreLingo course. It contains the definition of\na word. The word can be either in the source language or the target\nlanguage.\n\nDefinition in the source language (Spanish in this case)\n>>> DictionaryItem("hablo", "I speak", False)\nDictionaryItem(word=\'hablo\', definition=\'I speak\', is_in_target_language=False)\n\nDefinition in the target language (English in this case)\n>>> DictionaryItem("speak", "hablo", True)\nDictionaryItem(word=\'speak\', definition=\'hablo\', is_in_target_language=True)\n\n<a name="librelingo_types.data_types.PhraseIdentity"></a>\n## PhraseIdentity Objects\n\n```python\nclass PhraseIdentity(namedtuple("PhraseIdentity", ["text", "source"]))\n```\n\nThis is the set of information that identifies a phrase as \'the same\'. If any\nof these things change, the phrase will be seen as \'new\' and re-generated.\n\n',
    'author': 'Dániel Kántor',
    'author_email': 'git@daniel-kantor.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': None,
    'packages': packages,
    'package_data': package_data,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
