# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['space_packet_parser']

package_data = \
{'': ['*']}

install_requires = \
['bitstring>=3.0.0,<5']

setup_kwargs = {
    'name': 'space-packet-parser',
    'version': '4.0.1',
    'description': 'A CCSDS telemetry packet decoding library based on the XTCE packet format description standard.',
    'long_description': '# `Space Packet Parser`\nThis is a package for decoding CCSDS telemetry packets according to an XTCE or CSV format specification. \nIt is based on the UML model of the XTCE spec and aims to support all but the most esoteric elements of the \nXTCE telemetry packet specification.\n\nResources:\n- [XTCE (Green Book - Informational Report)](https://public.ccsds.org/Pubs/660x2g2.pdf)\n- [XTCE Element Description (Green Book - Informational Report)](https://public.ccsds.org/Pubs/660x1g2.pdf)\n- [XTCE (Blue Book - Recommended Standard)](https://public.ccsds.org/Pubs/660x0b2.pdf)\n\n## Installation\n```bash\npip install space_packet_parser\n```\n\n## Basic Usage\nUsage with XTCE packet definition:\n\n```python\nfrom pathlib import Path\nfrom space_packet_parser import xtcedef, parser\n\npacket_file = Path(\'my_packets.pkts\')\nxtce_document = Path(\'my_xtce_document.xml\')\npacket_definition = xtcedef.XtcePacketDefinition(xtce_document)\nmy_parser = parser.PacketParser(packet_definition)\n\nwith packet_file.open() as binary_data:\n    packet_generator = my_parser.generator(binary_data)\n\n    for packet in packet_generator:\n        # Do something with the packet data\n        print(packet.header[\'PKT_APID\'])\n        print(packet.data)\n```\n\nUsage with CSV packet definition:\n\n```python\nfrom pathlib import Path\nfrom space_packet_parser import csvdef, parser\n\npacket_file = Path(\'my_packets.pkts\')\ncsv_document = Path(\'my_csv_file.csv\')\npacket_definition = csvdef.CsvPacketDefinition(csv_document)\nmy_parser = parser.PacketParser(packet_definition)\n\nwith packet_file.open() as binary_data:\n    packet_generator = my_parser.generator(binary_data)\n\n    for packet in packet_generator:\n        # Do something with the packet data\n        print(packet.header[\'PKT_APID\'])\n        print(packet.data)\n```\n\n### Examples\nAs part of our integration testing, we have provided examples for several packet types.\nThe simplest is the JPSS parsing, which parses fixed length invariantly structured packets. \nThe most complex is SUDA, which uses a polymorphic packet structure \nbased on the value of `SCIFETCHTYPE` (for APID 1425) or `SCI0TYPE` (for APID 1424).\n\nThe only example for CSV parsing is CTIM.\n\n### Debugging Packet Parsing\nParsing binary packets is error-prone and getting the XTCE definition correct can be a challenge at first. \nMost flight software teams can export XTCE from their command and telemetry database but these exports usually require \nsome fine-tuning. \n\n`UnrecognizedPacketError`s are raised during parsing of an individual packet when either:\n\n- a) multiple child containers are valid inheritors of the current sequence container based on \n   restriction criteria evaluated against the data parsed so far\n- b) no child containers are valid inheritors of the current sequence container based on \n  restriction criteria evaluated against the data parsed so far \n  and the current container is abstract\n\nTo aid you during development, `UnrecognizedPacketError` exceptions generated during parsing can be returned \nalongside any valid packet objects by setting `yield_unrecognized_packet_errors=True`. \nThese exception objects are not raised so that the generator may keep parsing. Instead, they \nare yielded from the generator with a `partial_data` attribute for user examination. This partial data allows you to \nsee how far it got through a container inheritance structure before failing to determine the rest of the structure. \n\n### Optimizing for Performance\nThe logic evaluated during packet parsing is largely reflective of the XTCE configuration being used \nto define packet structures. The more logic in the XTCE, the more logic must be evaluated during \nparsing. Below are some common ways to reduce complexity and speed up parsing:\n\n1. **Remove `RestrictionCriteria` Elements:** If your packet stream is a single packet structure, \n    there is no reason to require the evaluation of a restriction criteria for each packet. \n2. **Remove Unnecessary Packet Definitions:** Even in a packet stream with multiple packet formats, if you only\n    care about one packet type, you can remove the definitions for the other. By default the packet `Parser` will \n    catch unrecognized packet errors and skip to the next packet. This skips the parsing of packets \n    for which a valid definition cannot be determined.\n3. **Reduce Container Inheritance:** A flat container definition structure will evaluate restriction criteria\n    faster than a nested structure. Each instance of nesting requires an additional `MatchCriteria.evaluate()`\n    call for each packet being parsed. \n4. **Reduce Complex Items:** Parameter type definitions that contain calibrators or complex string parsing \n    (especially variable length termination character defined strings) add significant evaluation logic to \n    the parsing of each parameter, as does any parameter type that is variable length. \n    Removing them can speed up parsing.\n\n## Developer Documentation\n### Installing Development Dependencies\nPoetry installs dev dependencies by default from the `poetry.lock` or `pyproject.toml` files. Just run \n```bash\npoetry install\n```\n\n### Testing\nTesting is run with `pytest` and the order is randomized by `pytest-randomly`. \nTo run all tests, run\n```bash\npytest tests\n```\n\nTo run all tests in docker containers (tests against many versions of python), run\n```bash\ndocker-compose up --build && docker-compose down\n```\n\n### Release Process\nReference: [https://www.atlassian.com/git/tutorials/comparing-workflows/gitflow-workflow]()\n\n1. Create a release candidate branch named according to the version to be released. This branch is used to polish\n   the release while work continues on dev (towards the next release). The naming convention is `release/X.Y.Z`\n\n2. Bump the version of the package to the version you are about to release, either manually by editing `pyproject.toml`\n   or by running `poetry version X.Y.Z` or bumping according to a valid bump rule like `poetry version minor`\n   (see poetry docs: https://python-poetry.org/docs/cli/#version).\n\n3. Update the version identifier in CITATION.cff.\n\n4. Update `CHANGES.md` to reflect that the version is now "released" and revisit `README.md` to keep it up to date.\n   \n5. Open a PR to merge the release branch into master. This informs the rest of the team how the release \n   process is progressing as you polish the release branch.\n\n6. When you are satisfied that the release branch is ready, merge the PR into `master`. \n\n7. Check out the `master` branch, pull the merged changes, and tag the newly created merge commit with the \n   desired version `X.Y.Z` and push the tag upstream. \n   \n   ```bash\n   git tag -a X.Y.Z -m "version release X.Y.Z"\n   git push origin X.Y.Z\n   ```\n   \n8. Ensure that you have `master` checked out and build the package (see below).\n   Check that the version of the built artifacts is as you expect (should match the version git tag and the \n   output from `poetry version --short`).\n   \n9. Optionally distribute the artifacts to PyPI/Nexus if desired (see below).\n   \n10. Open a PR to merge `master` back into `dev` so that any changes made during the release process are also captured\n    in `dev`. \n\n\n### Building and Distribution\n1. Ensure that `poetry` is installed by running `poetry --version`.\n   \n2. To build the distribution archives, run `poetry build`.\n   \n3. To upload the wheel to Nexus, run `poetry publish`. You will need credentials to sign into PyPI.\n',
    'author': 'Gavin Medley',
    'author_email': 'gavin.medley@lasp.colorado.edu',
    'maintainer': 'Gavin Medley',
    'maintainer_email': 'gavin.medley@lasp.colorado.edu',
    'url': 'https://github.com/medley56/space_packet_parser',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.8,<3.12',
}


setup(**setup_kwargs)
