"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.extractObjects = void 0;
const zlib_1 = require("zlib");
const tar_stream_1 = require("tar-stream");
/**
 * Extracts objects from a compressed tarball (.tgz file). Selectors are
 * provided as a map from result key to a selector configuration object, which
 * can specify whether the object is required (`required: true`) or not (a
 * missing required object will cause the extraction to throw an error), and
 * either a path name (exact match with the contents of the tarball), or a
 * filter function. Selector properties are applied in the order in which they
 * are defined: the first matching selector gets used to select the object. If
 * a selector matches multiple objects in the tarball, the last matched object
 * is returned.
 *
 * @param tgz      the compressed tarbal data in a `Buffer`.
 * @param selector the objects to be extracted from the tarball.
 *
 * @returns the extracted objects.
 */
async function extractObjects(tgz, selector) {
    const tarball = await gunzip(tgz);
    return new Promise((ok, ko) => {
        const result = {};
        const extractor = tar_stream_1.extract({ filenameEncoding: 'utf-8' })
            .once('error', ko)
            .once('finish', () => {
            for (const [name, { path, required }] of Object.entries(selector)) {
                if (!required) {
                    continue;
                }
                if (!(name in result)) {
                    const err = new Error(`Missing required entry in tarball: ${name} (${path !== null && path !== void 0 ? path : '<dynamic>'})`);
                    Error.captureStackTrace(err);
                    ko(err);
                    return;
                }
            }
            ok(result);
        })
            .on('entry', (headers, stream, next) => {
            for (const [name, config] of Object.entries(selector)) {
                if (selectorMatches(headers.name, config)) {
                    const chunks = new Array();
                    stream.once('error', ko)
                        .on('data', (chunk) => chunks.push(Buffer.from(chunk)))
                        .once('end', () => {
                        result[name] = Buffer.concat(chunks);
                        // Running `next` on the next runLoop iteration to avoid stack overflow
                        setImmediate(next);
                    })
                        .resume();
                    return;
                }
            }
            // Running `next` on the next runLoop iteration to avoid stack overflow
            setImmediate(next);
        });
        extractor.write(tarball, (err) => {
            if (err != null) {
                ko(err);
            }
            extractor.end();
        });
    });
}
exports.extractObjects = extractObjects;
function gunzip(gz) {
    return new Promise((ok, ko) => {
        const chunks = new Array();
        zlib_1.createGunzip()
            .once('error', ko)
            .on('data', chunk => chunks.push(Buffer.from(chunk)))
            .once('end', () => ok(Buffer.concat(chunks)))
            .end(gz);
    });
}
function selectorMatches(path, config) {
    if ('path' in config) {
        return path === config.path;
    }
    return config.filter(path);
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidGFyYmFsbC5sYW1iZGEtc2hhcmVkLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiLi4vLi4vLi4vc3JjL2JhY2tlbmQvc2hhcmVkL3RhcmJhbGwubGFtYmRhLXNoYXJlZC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7QUFBQSwrQkFBb0M7QUFDcEMsMkNBQXFDO0FBRXJDOzs7Ozs7Ozs7Ozs7Ozs7R0FlRztBQUNJLEtBQUssVUFBVSxjQUFjLENBQXFCLEdBQVcsRUFBRSxRQUFXO0lBQy9FLE1BQU0sT0FBTyxHQUFHLE1BQU0sTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDO0lBQ2xDLE9BQU8sSUFBSSxPQUFPLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUU7UUFDNUIsTUFBTSxNQUFNLEdBQStCLEVBQUUsQ0FBQztRQUM5QyxNQUFNLFNBQVMsR0FBRyxvQkFBTyxDQUFDLEVBQUUsZ0JBQWdCLEVBQUUsT0FBTyxFQUFFLENBQUM7YUFDckQsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUM7YUFDakIsSUFBSSxDQUFDLFFBQVEsRUFBRSxHQUFHLEVBQUU7WUFDbkIsS0FBSyxNQUFNLENBQUMsSUFBSSxFQUFFLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsRUFBRTtnQkFDakUsSUFBSSxDQUFDLFFBQVEsRUFBRTtvQkFDYixTQUFTO2lCQUNWO2dCQUNELElBQUksQ0FBQyxDQUFDLElBQUksSUFBSSxNQUFNLENBQUMsRUFBRTtvQkFDckIsTUFBTSxHQUFHLEdBQUcsSUFBSSxLQUFLLENBQUMsc0NBQXNDLElBQUksS0FBSyxJQUFJLGFBQUosSUFBSSxjQUFKLElBQUksR0FBSSxXQUFXLEdBQUcsQ0FBQyxDQUFDO29CQUM3RixLQUFLLENBQUMsaUJBQWlCLENBQUMsR0FBRyxDQUFDLENBQUM7b0JBQzdCLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQztvQkFDUixPQUFPO2lCQUNSO2FBQ0Y7WUFDRCxFQUFFLENBQUMsTUFBc0IsQ0FBQyxDQUFDO1FBQzdCLENBQUMsQ0FBQzthQUNELEVBQUUsQ0FBQyxPQUFPLEVBQUUsQ0FBQyxPQUFPLEVBQUUsTUFBTSxFQUFFLElBQUksRUFBRSxFQUFFO1lBQ3JDLEtBQUssTUFBTSxDQUFDLElBQUksRUFBRSxNQUFNLENBQUMsSUFBSSxNQUFNLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxFQUFFO2dCQUNyRCxJQUFJLGVBQWUsQ0FBQyxPQUFPLENBQUMsSUFBSSxFQUFFLE1BQU0sQ0FBQyxFQUFFO29CQUN6QyxNQUFNLE1BQU0sR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO29CQUNuQyxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRSxFQUFFLENBQUM7eUJBQ3JCLEVBQUUsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDO3lCQUN0RCxJQUFJLENBQUMsS0FBSyxFQUFFLEdBQUcsRUFBRTt3QkFDaEIsTUFBTSxDQUFDLElBQUksQ0FBQyxHQUFHLE1BQU0sQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUM7d0JBQ3JDLHVFQUF1RTt3QkFDdkUsWUFBWSxDQUFDLElBQUksQ0FBQyxDQUFDO29CQUNyQixDQUFDLENBQUM7eUJBQ0QsTUFBTSxFQUFFLENBQUM7b0JBQ1osT0FBTztpQkFDUjthQUNGO1lBQ0QsdUVBQXVFO1lBQ3ZFLFlBQVksQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNyQixDQUFDLENBQUMsQ0FBQztRQUNMLFNBQVMsQ0FBQyxLQUFLLENBQUMsT0FBTyxFQUFFLENBQUMsR0FBRyxFQUFFLEVBQUU7WUFDL0IsSUFBSSxHQUFHLElBQUksSUFBSSxFQUFFO2dCQUNmLEVBQUUsQ0FBQyxHQUFHLENBQUMsQ0FBQzthQUNUO1lBQ0QsU0FBUyxDQUFDLEdBQUcsRUFBRSxDQUFDO1FBQ2xCLENBQUMsQ0FBQyxDQUFDO0lBQ0wsQ0FBQyxDQUFDLENBQUM7QUFDTCxDQUFDO0FBN0NELHdDQTZDQztBQUVELFNBQVMsTUFBTSxDQUFDLEVBQVU7SUFDeEIsT0FBTyxJQUFJLE9BQU8sQ0FBQyxDQUFDLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRTtRQUM1QixNQUFNLE1BQU0sR0FBRyxJQUFJLEtBQUssRUFBVSxDQUFDO1FBQ25DLG1CQUFZLEVBQUU7YUFDWCxJQUFJLENBQUMsT0FBTyxFQUFFLEVBQUUsQ0FBQzthQUNqQixFQUFFLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxDQUFDLENBQUM7YUFDcEQsSUFBSSxDQUFDLEtBQUssRUFBRSxHQUFHLEVBQUUsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxNQUFNLENBQUMsQ0FBQyxDQUFDO2FBQzVDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUNiLENBQUMsQ0FBQyxDQUFDO0FBQ0wsQ0FBQztBQUVELFNBQVMsZUFBZSxDQUFDLElBQVksRUFBRSxNQUF3QjtJQUM3RCxJQUFJLE1BQU0sSUFBSSxNQUFNLEVBQUU7UUFDcEIsT0FBTyxJQUFJLEtBQUssTUFBTSxDQUFDLElBQUksQ0FBQztLQUM3QjtJQUNELE9BQU8sTUFBTSxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztBQUM3QixDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0IHsgY3JlYXRlR3VuemlwIH0gZnJvbSAnemxpYic7XG5pbXBvcnQgeyBleHRyYWN0IH0gZnJvbSAndGFyLXN0cmVhbSc7XG5cbi8qKlxuICogRXh0cmFjdHMgb2JqZWN0cyBmcm9tIGEgY29tcHJlc3NlZCB0YXJiYWxsICgudGd6IGZpbGUpLiBTZWxlY3RvcnMgYXJlXG4gKiBwcm92aWRlZCBhcyBhIG1hcCBmcm9tIHJlc3VsdCBrZXkgdG8gYSBzZWxlY3RvciBjb25maWd1cmF0aW9uIG9iamVjdCwgd2hpY2hcbiAqIGNhbiBzcGVjaWZ5IHdoZXRoZXIgdGhlIG9iamVjdCBpcyByZXF1aXJlZCAoYHJlcXVpcmVkOiB0cnVlYCkgb3Igbm90IChhXG4gKiBtaXNzaW5nIHJlcXVpcmVkIG9iamVjdCB3aWxsIGNhdXNlIHRoZSBleHRyYWN0aW9uIHRvIHRocm93IGFuIGVycm9yKSwgYW5kXG4gKiBlaXRoZXIgYSBwYXRoIG5hbWUgKGV4YWN0IG1hdGNoIHdpdGggdGhlIGNvbnRlbnRzIG9mIHRoZSB0YXJiYWxsKSwgb3IgYVxuICogZmlsdGVyIGZ1bmN0aW9uLiBTZWxlY3RvciBwcm9wZXJ0aWVzIGFyZSBhcHBsaWVkIGluIHRoZSBvcmRlciBpbiB3aGljaCB0aGV5XG4gKiBhcmUgZGVmaW5lZDogdGhlIGZpcnN0IG1hdGNoaW5nIHNlbGVjdG9yIGdldHMgdXNlZCB0byBzZWxlY3QgdGhlIG9iamVjdC4gSWZcbiAqIGEgc2VsZWN0b3IgbWF0Y2hlcyBtdWx0aXBsZSBvYmplY3RzIGluIHRoZSB0YXJiYWxsLCB0aGUgbGFzdCBtYXRjaGVkIG9iamVjdFxuICogaXMgcmV0dXJuZWQuXG4gKlxuICogQHBhcmFtIHRneiAgICAgIHRoZSBjb21wcmVzc2VkIHRhcmJhbCBkYXRhIGluIGEgYEJ1ZmZlcmAuXG4gKiBAcGFyYW0gc2VsZWN0b3IgdGhlIG9iamVjdHMgdG8gYmUgZXh0cmFjdGVkIGZyb20gdGhlIHRhcmJhbGwuXG4gKlxuICogQHJldHVybnMgdGhlIGV4dHJhY3RlZCBvYmplY3RzLlxuICovXG5leHBvcnQgYXN5bmMgZnVuY3Rpb24gZXh0cmFjdE9iamVjdHM8UyBleHRlbmRzIFNlbGVjdG9yPih0Z3o6IEJ1ZmZlciwgc2VsZWN0b3I6IFMpOiBQcm9taXNlPFNlbGVjdGlvbjxTPj4ge1xuICBjb25zdCB0YXJiYWxsID0gYXdhaXQgZ3VuemlwKHRneik7XG4gIHJldHVybiBuZXcgUHJvbWlzZSgob2ssIGtvKSA9PiB7XG4gICAgY29uc3QgcmVzdWx0OiB7IFtuYW1lOiBzdHJpbmddOiBCdWZmZXIgfSA9IHt9O1xuICAgIGNvbnN0IGV4dHJhY3RvciA9IGV4dHJhY3QoeyBmaWxlbmFtZUVuY29kaW5nOiAndXRmLTgnIH0pXG4gICAgICAub25jZSgnZXJyb3InLCBrbylcbiAgICAgIC5vbmNlKCdmaW5pc2gnLCAoKSA9PiB7XG4gICAgICAgIGZvciAoY29uc3QgW25hbWUsIHsgcGF0aCwgcmVxdWlyZWQgfV0gb2YgT2JqZWN0LmVudHJpZXMoc2VsZWN0b3IpKSB7XG4gICAgICAgICAgaWYgKCFyZXF1aXJlZCkge1xuICAgICAgICAgICAgY29udGludWU7XG4gICAgICAgICAgfVxuICAgICAgICAgIGlmICghKG5hbWUgaW4gcmVzdWx0KSkge1xuICAgICAgICAgICAgY29uc3QgZXJyID0gbmV3IEVycm9yKGBNaXNzaW5nIHJlcXVpcmVkIGVudHJ5IGluIHRhcmJhbGw6ICR7bmFtZX0gKCR7cGF0aCA/PyAnPGR5bmFtaWM+J30pYCk7XG4gICAgICAgICAgICBFcnJvci5jYXB0dXJlU3RhY2tUcmFjZShlcnIpO1xuICAgICAgICAgICAga28oZXJyKTtcbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgb2socmVzdWx0IGFzIFNlbGVjdGlvbjxTPik7XG4gICAgICB9KVxuICAgICAgLm9uKCdlbnRyeScsIChoZWFkZXJzLCBzdHJlYW0sIG5leHQpID0+IHtcbiAgICAgICAgZm9yIChjb25zdCBbbmFtZSwgY29uZmlnXSBvZiBPYmplY3QuZW50cmllcyhzZWxlY3RvcikpIHtcbiAgICAgICAgICBpZiAoc2VsZWN0b3JNYXRjaGVzKGhlYWRlcnMubmFtZSwgY29uZmlnKSkge1xuICAgICAgICAgICAgY29uc3QgY2h1bmtzID0gbmV3IEFycmF5PEJ1ZmZlcj4oKTtcbiAgICAgICAgICAgIHN0cmVhbS5vbmNlKCdlcnJvcicsIGtvKVxuICAgICAgICAgICAgICAub24oJ2RhdGEnLCAoY2h1bmspID0+IGNodW5rcy5wdXNoKEJ1ZmZlci5mcm9tKGNodW5rKSkpXG4gICAgICAgICAgICAgIC5vbmNlKCdlbmQnLCAoKSA9PiB7XG4gICAgICAgICAgICAgICAgcmVzdWx0W25hbWVdID0gQnVmZmVyLmNvbmNhdChjaHVua3MpO1xuICAgICAgICAgICAgICAgIC8vIFJ1bm5pbmcgYG5leHRgIG9uIHRoZSBuZXh0IHJ1bkxvb3AgaXRlcmF0aW9uIHRvIGF2b2lkIHN0YWNrIG92ZXJmbG93XG4gICAgICAgICAgICAgICAgc2V0SW1tZWRpYXRlKG5leHQpO1xuICAgICAgICAgICAgICB9KVxuICAgICAgICAgICAgICAucmVzdW1lKCk7XG4gICAgICAgICAgICByZXR1cm47XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIC8vIFJ1bm5pbmcgYG5leHRgIG9uIHRoZSBuZXh0IHJ1bkxvb3AgaXRlcmF0aW9uIHRvIGF2b2lkIHN0YWNrIG92ZXJmbG93XG4gICAgICAgIHNldEltbWVkaWF0ZShuZXh0KTtcbiAgICAgIH0pO1xuICAgIGV4dHJhY3Rvci53cml0ZSh0YXJiYWxsLCAoZXJyKSA9PiB7XG4gICAgICBpZiAoZXJyICE9IG51bGwpIHtcbiAgICAgICAga28oZXJyKTtcbiAgICAgIH1cbiAgICAgIGV4dHJhY3Rvci5lbmQoKTtcbiAgICB9KTtcbiAgfSk7XG59XG5cbmZ1bmN0aW9uIGd1bnppcChnejogQnVmZmVyKTogUHJvbWlzZTxCdWZmZXI+IHtcbiAgcmV0dXJuIG5ldyBQcm9taXNlKChvaywga28pID0+IHtcbiAgICBjb25zdCBjaHVua3MgPSBuZXcgQXJyYXk8QnVmZmVyPigpO1xuICAgIGNyZWF0ZUd1bnppcCgpXG4gICAgICAub25jZSgnZXJyb3InLCBrbylcbiAgICAgIC5vbignZGF0YScsIGNodW5rID0+IGNodW5rcy5wdXNoKEJ1ZmZlci5mcm9tKGNodW5rKSkpXG4gICAgICAub25jZSgnZW5kJywgKCkgPT4gb2soQnVmZmVyLmNvbmNhdChjaHVua3MpKSlcbiAgICAgIC5lbmQoZ3opO1xuICB9KTtcbn1cblxuZnVuY3Rpb24gc2VsZWN0b3JNYXRjaGVzKHBhdGg6IHN0cmluZywgY29uZmlnOiBTZWxlY3RvclByb3BlcnR5KTogYm9vbGVhbiB7XG4gIGlmICgncGF0aCcgaW4gY29uZmlnKSB7XG4gICAgcmV0dXJuIHBhdGggPT09IGNvbmZpZy5wYXRoO1xuICB9XG4gIHJldHVybiBjb25maWcuZmlsdGVyKHBhdGgpO1xufVxuXG5pbnRlcmZhY2UgU2VsZWN0b3Ige1xuICByZWFkb25seSBbbmFtZTogc3RyaW5nXTogU2VsZWN0b3JQcm9wZXJ0eTtcbn1cbnR5cGUgU2VsZWN0b3JQcm9wZXJ0eSA9IHsgcmVhZG9ubHkgcmVxdWlyZWQ/OiBib29sZWFuIH1cbiYgKHsgcmVhZG9ubHkgcGF0aDogc3RyaW5nIH0gfCB7IHJlYWRvbmx5IHBhdGg/OiB1bmRlZmluZWQ7IHJlYWRvbmx5IGZpbHRlcjogKHBhdGg6IHN0cmluZykgPT4gYm9vbGVhbiB9KTtcblxudHlwZSBTZWxlY3Rpb248UyBleHRlbmRzIFNlbGVjdG9yPiA9IHtcbiAgcmVhZG9ubHkgW1AgaW4ga2V5b2YgU106IFNbUF0gZXh0ZW5kcyB7IHJlYWRvbmx5IHJlcXVpcmVkOiB0cnVlIH0gPyBCdWZmZXIgOiBCdWZmZXIgfCB1bmRlZmluZWQ7XG59O1xuIl19