"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.codeArtifactPublishPackage = exports.logInWithCodeArtifact = void 0;
const os_1 = require("os");
const path_1 = require("path");
const aws_sdk_1 = require("aws-sdk");
const fs_extra_1 = require("fs-extra");
const shell_out_lambda_shared_1 = require("./shell-out.lambda-shared");
/**
 * Logs into the provided CodeArtifact registry, and makes it the default NPM
 * registry for this environment.
 */
async function logInWithCodeArtifact({ endpoint, domain, domainOwner, apiEndpoint }) {
    // Remove the protocol part of the endpoint URL, keeping the rest intact.
    const protoRelativeEndpoint = endpoint.replace(/^[^:]+:/, '');
    const { authorizationToken } = await new aws_sdk_1.CodeArtifact({ endpoint: apiEndpoint }).getAuthorizationToken({
        domain,
        domainOwner,
        durationSeconds: 0,
    }).promise();
    await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `registry=${endpoint}`);
    await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `${protoRelativeEndpoint}:_authToken=${authorizationToken}`);
    await shell_out_lambda_shared_1.shellOut('npm', 'config', 'set', `${protoRelativeEndpoint}:always-auth=true`);
}
exports.logInWithCodeArtifact = logInWithCodeArtifact;
/**
 * Publishes the provided tarball to the specified CodeArtifact repository.
 *
 * @param tarball a Buffer containing the tarball for the published package.
 * @param opts    the informations about the CodeArtifact repository.
 */
async function codeArtifactPublishPackage(tarball, opts) {
    var _a, _b;
    // Working in a temporary directory, so we can log into CodeArtifact and not leave traces.
    const cwd = await fs_extra_1.mkdtemp(path_1.join(os_1.tmpdir(), 'npm-publish-'));
    const oldHome = process.env.HOME;
    try {
        process.env.HOME = cwd;
        await logInWithCodeArtifact(opts);
        const tarballPath = path_1.join(cwd, 'tarball.tgz');
        await fs_extra_1.writeFile(tarballPath, tarball);
        const { exitCode, signal, stdout } = await shell_out_lambda_shared_1.shellOutWithOutput('npm', 'publish', '--json', tarballPath);
        if (exitCode === 0) {
            return;
        }
        if (signal != null) {
            throw new Error(`npm publish was killed by signal ${signal}`);
        }
        const result = JSON.parse(stdout.toString('utf-8'));
        if (((_a = result.error) === null || _a === void 0 ? void 0 : _a.code) === 'E409' || ((_b = result.error) === null || _b === void 0 ? void 0 : _b.code) === 'EPUBLISHCONFLICT') {
            console.log(`${result.error.code} - The package already exist; assuming idempotent success!`);
            return;
        }
        else {
            throw new Error(`npm publish returned ${JSON.stringify(result)}`);
        }
    }
    finally {
        // Restore the previous environment, and remove temporary directory
        process.env.HOME = oldHome;
        await fs_extra_1.remove(cwd);
    }
}
exports.codeArtifactPublishPackage = codeArtifactPublishPackage;
//# sourceMappingURL=data:application/json;base64,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