"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.NpmDownloadsClient = exports.NpmDownloadsPeriod = void 0;
var NpmDownloadsPeriod;
(function (NpmDownloadsPeriod) {
    /**
     * Gets downloads for the last available day. In practice, this will usually
     * be "yesterday" (in GMT) but if stats for that day have not yet landed, it
     * will be the day before.
     */
    NpmDownloadsPeriod["LAST_DAY"] = "last-day";
    /**
     * Gets downloads for the last 7 available days.
     */
    NpmDownloadsPeriod["LAST_WEEK"] = "last-week";
    /**
     * Gets downloads for the last 30 available days.
     */
    NpmDownloadsPeriod["LAST_MONTH"] = "last-month";
})(NpmDownloadsPeriod = exports.NpmDownloadsPeriod || (exports.NpmDownloadsPeriod = {}));
class NpmDownloadsClient {
    constructor(gotService) {
        this.got = gotService;
    }
    async getDownloadsRaw(packages, period, throwErrors) {
        if (packages.length > NpmDownloadsClient.MAX_PACKAGES_PER_QUERY) {
            throw new Error(`Too many packages were provided (max: ${NpmDownloadsClient.MAX_PACKAGES_PER_QUERY})`);
        }
        if (packages.some((pkg) => this.isScopedPackage(pkg)) && packages.length > 1) {
            throw new Error('Scoped packages aren\'t supported by the bulk query API.');
        }
        if (packages.length === 0)
            return new Map();
        console.log(`Querying NPM for ${packages.length} package(s): [${packages.join(', ')}]`);
        const result = await this.got(`${NpmDownloadsClient.NPM_DOWNLOADS_API_URL}/${period}/${packages.join(',')}`, {
            timeout: 5 * 1000,
        }).catch((err) => {
            if (throwErrors) {
                throw err;
            }
            else {
                return { body: JSON.stringify({ error: JSON.stringify(err) }) };
            }
        });
        const data = JSON.parse(result.body);
        // single package query error
        // ex. { "error": "package foo not found" }
        if ('error' in data) {
            if (throwErrors) {
                throw new Error(`Could not retrieve download metrics: ${data.error}`);
            }
            else {
                console.error(`Could not retrieve download metrics: ${data.error}`);
                return new Map();
            }
        }
        // only a single package was returned
        if (isSingleDownloadsEntry(data)) {
            return new Map([[packages[0], data]]);
        }
        // bulk query result
        for (const key of Object.keys(data)) {
            if (data[key] === null) {
                if (throwErrors) {
                    throw new Error(`Could not retrieve download metrics for package ${key}`);
                }
                else {
                    console.error(`Could not retrieve download metrics for package ${key}`);
                    delete data[key];
                }
            }
        }
        // typescript can't figure out that we removed all null values
        // @ts-ignore
        return new Map(Object.entries(data));
    }
    /**
     * Retrieves the number of downloads each package has on npm in the latest period.
     * Output is not guaranteed to be returned in a specific order.
     * If throwErrors option is specified, an error is thrown when a package's
     * download count is unavailable - otherwise, it's just omitted from
     * the output.
     */
    async getDownloads(packages, options = {}) {
        var _a, _b;
        const period = (_a = options.period) !== null && _a !== void 0 ? _a : NpmDownloadsPeriod.LAST_WEEK;
        const throwErrors = (_b = options.throwErrors) !== null && _b !== void 0 ? _b : true;
        // separate scoped and unscoped packages since scoped packages are not
        // supported by the bulk query API
        const scopedPackages = [];
        const unscopedPackages = [];
        for (const pkg of packages) {
            if (this.isScopedPackage(pkg)) {
                scopedPackages.push(pkg);
            }
            else {
                unscopedPackages.push(pkg);
            }
        }
        // we could parallelize this, but then it's more likely we get throttled
        const output = new Map();
        for (const pkg of scopedPackages) {
            const partialResults = await this.getDownloadsRaw([pkg], period, throwErrors);
            for (const [key, value] of partialResults) {
                output.set(key, value);
            }
        }
        for (let i = 0; i < unscopedPackages.length; i += NpmDownloadsClient.MAX_PACKAGES_PER_QUERY) {
            const batch = unscopedPackages.slice(i, i + NpmDownloadsClient.MAX_PACKAGES_PER_QUERY);
            const partialResults = await this.getDownloadsRaw(batch, period, throwErrors);
            for (const [key, value] of partialResults) {
                output.set(key, value);
            }
        }
        return output;
    }
    isScopedPackage(packageName) {
        return packageName.startsWith('@');
    }
}
exports.NpmDownloadsClient = NpmDownloadsClient;
NpmDownloadsClient.NPM_DOWNLOADS_API_URL = 'https://api.npmjs.org/downloads/point';
NpmDownloadsClient.MAX_PACKAGES_PER_QUERY = 128; // this is a limitation from npm's API! do not change
function isSingleDownloadsEntry(data) {
    return 'downloads' in data && typeof data.downloads === 'number';
}
//# sourceMappingURL=data:application/json;base64,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