"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LicenseList = void 0;
const fs = require("fs");
const os = require("os");
const path = require("path");
const s3 = require("@aws-cdk/aws-s3");
const s3deploy = require("@aws-cdk/aws-s3-deployment");
const core_1 = require("@aws-cdk/core");
/**
 * A list of licenses, which can be used to control filter packages before
 * indexing.
 */
class LicenseList extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The object key in which the license list is stored.
         */
        this.objectKey = 'allowed-licenses.json';
        this.bucket = new s3.Bucket(this, 'Bucket', {
            blockPublicAccess: s3.BlockPublicAccess.BLOCK_ALL,
            encryption: s3.BucketEncryption.S3_MANAGED,
            enforceSSL: true,
            removalPolicy: core_1.RemovalPolicy.DESTROY,
            versioned: true,
        });
        this.upload = new s3deploy.BucketDeployment(this, 'Resource', {
            destinationBucket: this.bucket,
            prune: true,
            retainOnDelete: false,
            sources: [this.createAsset(props.licenses)],
        });
    }
    /**
     * Grants an AWS Lambda function permissions to read the license allow list,
     * and adds the relevant environment variables expected by the
     * `LicenseListClient`.
     */
    grantRead(handler) {
        handler.addEnvironment("LICENSE_LIST_BUCKET_NAME" /* BUCKET_NAME */, this.bucket.bucketName);
        handler.addEnvironment("LICENSE_LIST_OBJECT_KEY" /* OBJECT_KEY */, this.objectKey);
        this.bucket.grantRead(handler);
        // The handler now depends on the deny-list having been uploaded
        handler.node.addDependency(this.upload);
    }
    createAsset(licenses) {
        const tmpdir = fs.mkdtempSync(path.join(os.tmpdir(), 'license-list-'));
        fs.writeFileSync(path.join(tmpdir, this.objectKey), JSON.stringify(licenses.map((l) => l.id), null, 2), 'utf-8');
        return s3deploy.Source.asset(tmpdir);
    }
}
exports.LicenseList = LicenseList;
//# sourceMappingURL=data:application/json;base64,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