"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const aws_embedded_metrics_1 = require("aws-embedded-metrics");
const semver_1 = require("semver");
const aws = require("../shared/aws.lambda-shared");
const compress_content_lambda_shared_1 = require("../shared/compress-content.lambda-shared");
const constants = require("../shared/constants");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
const language_1 = require("../shared/language");
const constants_1 = require("./constants");
aws_embedded_metrics_1.Configuration.namespace = constants_1.METRICS_NAMESPACE;
async function handler(event, context) {
    var _a;
    console.log('Event:', JSON.stringify(event, null, 2));
    const indexedPackages = new Map();
    const packageNames = new Set();
    const packageMajorVersions = new Set();
    const perLanguage = new Map();
    /**
     * Records the status of a particular package, package major version, package
     * version, and package version submodule in the per-language state storage.
     * Whenever a new entry is added, a `MISSING` entry is automatically inserted
     * for the other languages (unless another entry already exists).
     *
     * If a submodule is provided, only that submodule's availability is updated.
     */
    function recordPerLanguage(language, status, pkgName, pkgMajor, pkgVersion, submodule) {
        for (const lang of language_1.DocumentationLanguage.ALL) {
            doRecordPerLanguage(perLanguage, lang, 
            // If the language is NOT the registered one, then we insert "MISSING".
            lang === language ? status : "Missing" /* MISSING */, pkgName, pkgMajor, pkgVersion, submodule);
        }
    }
    const bucket = env_lambda_shared_1.requireEnv('BUCKET_NAME');
    for await (const key of relevantObjectKeys(bucket)) {
        const [, name, version] = constants.STORAGE_KEY_FORMAT_REGEX.exec(key);
        packageNames.add(name);
        const majorVersion = `${name}@${new semver_1.SemVer(version).major}`;
        packageMajorVersions.add(majorVersion);
        const fullName = `${name}@${version}`;
        // Ensure the package is fully registered for per-language status, even if no doc exists yet.
        for (const language of language_1.DocumentationLanguage.ALL) {
            recordPerLanguage(language, "Missing" /* MISSING */, name, majorVersion, fullName);
        }
        if (!indexedPackages.has(fullName)) {
            indexedPackages.set(fullName, {});
        }
        const status = indexedPackages.get(fullName);
        if (key.endsWith(constants.METADATA_KEY_SUFFIX)) {
            status.metadataPresent = true;
        }
        else if (key.endsWith(constants.PACKAGE_KEY_SUFFIX)) {
            status.tarballPresent = true;
        }
        else if (key.endsWith(constants.ASSEMBLY_KEY_SUFFIX)) {
            status.assemblyPresent = true;
        }
        else {
            let identified = false;
            for (const language of language_1.DocumentationLanguage.ALL) {
                const match = submoduleKeyRegexp(language).exec(key);
                if (match != null) {
                    const [, submodule, isUnsupported] = match;
                    if (status.submodules == null) {
                        status.submodules = new Set();
                    }
                    status.submodules.add(`${fullName}.${submodule}`);
                    recordPerLanguage(language, isUnsupported ? "Unsupported" /* UNSUPPORTED */ : "Supported" /* SUPPORTED */, name, majorVersion, fullName, submodule);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language))) {
                    recordPerLanguage(language, "Supported" /* SUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
                else if (key.endsWith(constants.docsKeySuffix(language) + constants.NOT_SUPPORTED_SUFFIX)) {
                    recordPerLanguage(language, "Unsupported" /* UNSUPPORTED */, name, majorVersion, fullName);
                    identified = true;
                }
            }
            if (!identified) {
                status.unknownObjects = (_a = status.unknownObjects) !== null && _a !== void 0 ? _a : [];
                status.unknownObjects.push(key);
            }
        }
    }
    await aws_embedded_metrics_1.metricScope((metrics) => () => {
        var _a, _b, _c;
        // Clear out default dimensions as we don't need those. See https://github.com/awslabs/aws-embedded-metrics-node/issues/73.
        metrics.setDimensions();
        const missingMetadata = new Array();
        const missingAssembly = new Array();
        const missingTarball = new Array();
        const unknownObjects = new Array();
        const submodules = new Array();
        for (const [name, status] of indexedPackages.entries()) {
            if (!status.metadataPresent) {
                missingMetadata.push(name);
            }
            if (!status.assemblyPresent) {
                missingAssembly.push(name);
            }
            if (!status.tarballPresent) {
                missingTarball.push(name);
            }
            if ((_b = (_a = status.unknownObjects) === null || _a === void 0 ? void 0 : _a.length) !== null && _b !== void 0 ? _b : 0 > 0) {
                unknownObjects.push(...status.unknownObjects);
            }
            for (const submodule of (_c = status.submodules) !== null && _c !== void 0 ? _c : []) {
                submodules.push(submodule);
            }
        }
        metrics.setProperty('detail', { missingMetadata, missingAssembly, missingTarball, unknownObjects });
        metrics.putMetric("MissingPackageMetadataCount" /* MISSING_METADATA_COUNT */, missingMetadata.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingAssemblyCount" /* MISSING_ASSEMBLY_COUNT */, missingAssembly.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("MissingPackageTarballCount" /* MISSING_TARBALL_COUNT */, missingTarball.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageCount" /* PACKAGE_COUNT */, packageNames.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageMajorVersionCount" /* PACKAGE_MAJOR_COUNT */, packageMajorVersions.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("PackageVersionCount" /* PACKAGE_VERSION_COUNT */, indexedPackages.size, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("SubmoduleCount" /* SUBMODULE_COUNT */, submodules.length, aws_embedded_metrics_1.Unit.Count);
        metrics.putMetric("UnknownObjectCount" /* UNKNOWN_OBJECT_COUNT */, unknownObjects.length, aws_embedded_metrics_1.Unit.Count);
    })();
    for (const entry of Array.from(perLanguage.entries())) {
        await aws_embedded_metrics_1.metricScope((metrics) => async (language, data) => {
            console.log('');
            console.log('##################################################');
            console.log(`### Start of data for ${language}`);
            metrics.setDimensions({ [constants_1.LANGUAGE_DIMENSION]: language.toString() });
            for (const forStatus of ["Supported" /* SUPPORTED */, "Unsupported" /* UNSUPPORTED */, "Missing" /* MISSING */]) {
                for (const [key, statuses] of Object.entries(data)) {
                    let filtered = Array.from(statuses.entries()).filter(([, status]) => forStatus === status);
                    let metricName = METRIC_NAME_BY_STATUS_AND_GRAIN[forStatus][key];
                    if (forStatus === "Missing" /* MISSING */) {
                        // Creates an object in S3 with the outcome of the scan for this language.
                        const { buffer, contentEncoding } = compress_content_lambda_shared_1.compressContent(Buffer.from(JSON.stringify(filtered.map(([name]) => name).sort(), null, 2)));
                        const missingDocsKey = constants.missingDocumentationKey(language);
                        console.log(`Uploading missing documentation list to s3://${bucket}/${missingDocsKey}`);
                        await aws.s3().putObject({
                            Body: buffer,
                            Bucket: bucket,
                            ContentEncoding: contentEncoding,
                            ContentType: 'application/json',
                            Expires: new Date(Date.now() + 300000),
                            Key: missingDocsKey,
                            Metadata: {
                                'Lambda-Run-Id': context.awsRequestId,
                                'Lambda-Log-Group-Name': context.logGroupName,
                                'Lambda-Log-Stream-Name': context.logStreamName,
                            },
                        }).promise();
                    }
                    console.log(`${forStatus} ${key} for ${language}: ${filtered.length} entries`);
                    metrics.putMetric(metricName, filtered.length, aws_embedded_metrics_1.Unit.Count);
                }
            }
            console.log(`### End of data for ${language}`);
            console.log('##################################################');
            console.log('');
        })(...entry);
    }
}
exports.handler = handler;
async function* relevantObjectKeys(bucket) {
    var _a;
    const request = {
        Bucket: bucket,
        Prefix: constants.STORAGE_KEY_PREFIX,
    };
    do {
        const response = await aws.s3().listObjectsV2(request).promise();
        for (const { Key } of (_a = response.Contents) !== null && _a !== void 0 ? _a : []) {
            if (Key == null) {
                continue;
            }
            yield Key;
        }
        request.ContinuationToken = response.NextContinuationToken;
    } while (request.ContinuationToken != null);
}
/**
 * This function obtains a regular expression with a capture group that allows
 * determining the submodule name from a submodule documentation key, and
 * another to determine whether the object is an "unsupported beacon" or not.
 */
function submoduleKeyRegexp(language) {
    // We use a placeholder to be able to insert the capture group once we have
    // fully quoted the key prefix for Regex safety.
    const placeholder = '<SUBMODULENAME>';
    // We obtain the standard key prefix.
    const keyPrefix = constants.docsKeySuffix(language, placeholder);
    // Finally, assemble the regular expression with the capture group.
    return new RegExp(`.*${reQuote(keyPrefix).replace(placeholder, '(.+)')}(${reQuote(constants.NOT_SUPPORTED_SUFFIX)})?$`);
    /**
     * Escapes all "speacial meaning" characters in a string, so it can be used as
     * part of a regular expression.
     */
    function reQuote(str) {
        return str.replace(/([+*.()?$[\]])/g, '\\$1');
    }
}
const METRIC_NAME_BY_STATUS_AND_GRAIN = {
    ["Missing" /* MISSING */]: {
        ["packages" /* PACKAGES */]: "MissingPackageCount" /* PER_LANGUAGE_MISSING_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "MissingMajorVersionCount" /* PER_LANGUAGE_MISSING_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "MissingPackageVersionCount" /* PER_LANGUAGE_MISSING_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "MissingSubmoduleCount" /* PER_LANGUAGE_MISSING_SUBMODULES */,
    },
    ["Unsupported" /* UNSUPPORTED */]: {
        ["packages" /* PACKAGES */]: "UnsupportedPackageCount" /* PER_LANGUAGE_UNSUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "UnsupportedMajorVersionCount" /* PER_LANGUAGE_UNSUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "UnsupportedPackageVersionCount" /* PER_LANGUAGE_UNSUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "UnsupportedSubmoduleCount" /* PER_LANGUAGE_UNSUPPORTED_SUBMODULES */,
    },
    ["Supported" /* SUPPORTED */]: {
        ["packages" /* PACKAGES */]: "SupportedPackageCount" /* PER_LANGUAGE_SUPPORTED_PACKAGES */,
        ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: "SupportedMajorVersionCount" /* PER_LANGUAGE_SUPPORTED_MAJORS */,
        ["package versions" /* PACKAGE_VERSIONS */]: "SupportedPackageVersionCount" /* PER_LANGUAGE_SUPPORTED_VERSIONS */,
        ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: "SupportedSubmoduleCount" /* PER_LANGUAGE_SUPPORTED_SUBMODULES */,
    },
};
/**
 * Registers the information for the provided language. A "MISSING" status
 * will be ignored if another status was already registered for the same
 * entity. An "UNSUPPORTED" status will be ignored if a "SUPPORTED" status
 * was already registered for the same entity.
 *
 * If a submodule is provided, only that submodule's availability is updated.
 */
function doRecordPerLanguage(perLanguage, language, status, pkgName, pkgMajor, pkgVersion, submodule) {
    if (!perLanguage.has(language)) {
        perLanguage.set(language, {
            ["package major versions" /* PACKAGE_MAJOR_VERSIONS */]: new Map(),
            ["packages" /* PACKAGES */]: new Map(),
            ["package version submodules" /* PACKAGE_VERSION_SUBMODULES */]: new Map(),
            ["package versions" /* PACKAGE_VERSIONS */]: new Map(),
        });
    }
    const data = perLanguage.get(language);
    // If there is a submodule, only update the submodule domain.
    const outputDomains = submodule
        ? [
            [data["package version submodules" /* PACKAGE_VERSION_SUBMODULES */], `${pkgVersion}.${submodule}`],
        ]
        : [
            [data["package major versions" /* PACKAGE_MAJOR_VERSIONS */], pkgMajor],
            [data["package versions" /* PACKAGE_VERSIONS */], pkgVersion],
            [data["packages" /* PACKAGES */], pkgName],
        ];
    for (const [map, name] of outputDomains) {
        switch (status) {
            case "Missing" /* MISSING */:
                // If we already have a status, don't override it with "MISSING".
                if (!map.has(name)) {
                    map.set(name, status);
                }
                break;
            case "Supported" /* SUPPORTED */:
                // If thr package is "supported", this always "wins"
                map.set(name, status);
                break;
            case "Unsupported" /* UNSUPPORTED */:
                // If we already have a status, only override with "UNSUPPORTED" if it was "MISSING".
                if (!map.has(name) || map.get(name) === "Missing" /* MISSING */) {
                    map.set(name, status);
                }
                break;
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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