"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Prune = void 0;
const aws_lambda_event_sources_1 = require("@aws-cdk/aws-lambda-event-sources");
const sqs = require("@aws-cdk/aws-sqs");
const core_1 = require("@aws-cdk/core");
const constants_1 = require("./constants");
const prune_handler_1 = require("./prune-handler");
const prune_queue_handler_1 = require("./prune-queue-handler");
/**
 * Reads the deny list and prunes all objects from the package data bucket
 * related to packages that match one of the deny list rules.
 */
class Prune extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // invoke the prune handler every time the deny list is updated.
        const deleteQueue = new sqs.Queue(this, 'DeleteQueue', {
            visibilityTimeout: core_1.Duration.minutes(2),
        });
        // this handler reads the deny list and queues all the matched objects for
        // deletion into the prune queue.
        const pruneHandler = new prune_handler_1.PruneHandler(this, 'PruneHandler', {
            timeout: core_1.Duration.minutes(15),
            environment: {
                [constants_1.ENV_PRUNE_PACKAGE_DATA_BUCKET_NAME]: props.packageDataBucket.bucketName,
                [constants_1.ENV_PRUNE_PACKAGE_DATA_KEY_PREFIX]: props.packageDataKeyPrefix,
                [constants_1.ENV_PRUNE_QUEUE_URL]: deleteQueue.queueUrl,
            },
        });
        deleteQueue.grantSendMessages(pruneHandler);
        props.packageDataBucket.grantRead(pruneHandler);
        // processes messages. each message includes an object key to delete.
        const deleteHandler = new prune_queue_handler_1.PruneQueueHandler(this, 'PruneQueueHandler', {
            timeout: core_1.Duration.minutes(1),
            environment: {
                [constants_1.ENV_DELETE_OBJECT_DATA_BUCKET_NAME]: props.packageDataBucket.bucketName,
            },
        });
        props.packageDataBucket.grantDelete(deleteHandler);
        deleteHandler.addEventSource(new aws_lambda_event_sources_1.SqsEventSource(deleteQueue)); // reads from the queue
        this.pruneHandler = pruneHandler;
        this.queue = deleteQueue;
        this.deleteHandler = deleteHandler;
        props.monitoring.watchful.watchLambdaFunction('Deny List - Prune Function', this.pruneHandler);
        props.monitoring.watchful.watchLambdaFunction('Deny List - Prune Delete Function', this.deleteHandler);
    }
    /**
     * Should be called to rebuild the catalog when the deny list changes.
     */
    onChangeInvoke(callback) {
        callback.grantInvoke(this.pruneHandler);
        this.pruneHandler.addEnvironment(constants_1.ENV_PRUNE_ON_CHANGE_FUNCTION_NAME, callback.functionArn);
    }
}
exports.Prune = Prune;
//# sourceMappingURL=data:application/json;base64,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