"""Objects abstracting UserDetailList, GroupDetailList, or RoleDetailList sections of the output generated by the
command `aws iam get-account-authorization-details`"""
# Copyright (c) 2020, salesforce.com, inc.
# All rights reserved.
# Licensed under the BSD 3-Clause license.
# For full license text, see the LICENSE file in the repo root
# or https://opensource.org/licenses/BSD-3-Clause
import logging
from policy_sentry.util.arns import get_account_from_arn, get_resource_path_from_arn
from cloudsplaining.shared.exclusions import is_name_excluded
from cloudsplaining.scan.policy_document import PolicyDocument
from cloudsplaining.shared.constants import DEFAULT_EXCLUSIONS_CONFIG
from cloudsplaining.scan.assume_role_policy_document import (
    AssumeRolePolicyDocument,
)

logger = logging.getLogger(__name__)


class PrincipalTypeDetails:
    """This covers UserDetailList, GroupDetailList, or RoleDetailList"""

    def __init__(self, principal_type_details):
        self.principals = []
        for principal_details in principal_type_details:
            self.principals.append(Principal(principal_details))


# pylint: disable=too-many-instance-attributes
class Principal:
    """
    Holds data for individual Principal Entries from the Authz file
    """

    def __init__(self, principal_detail):
        self.principal_detail = principal_detail
        if "UserName" in principal_detail:
            self.principal_type = "User"
        elif "GroupName" in principal_detail:
            self.principal_type = "Group"
        elif "RoleName" in principal_detail:
            self.principal_type = "Role"

        self.principal_policy_list = self._principal_policies()
        self.policy_list = []
        self.path = principal_detail.get("Path", None)
        self.arn = principal_detail.get("Arn", None)
        self.tags = principal_detail.get("Tags", None)
        self.create_date = principal_detail.get("CreateDate", None)
        # This will only appear for Roles, not Users or Groups
        self.assume_role_policy_document = self._assume_role_policy_document()
        # self.assume_role_policy_document = principal_detail.get("AssumeRolePolicyDocument", None)
        self.attached_managed_policies = principal_detail.get(
            "AttachedManagedPolicies", None
        )
        self.principal_policies = self._principal_policies()
        if self.principal_type == "User":
            self.name = principal_detail.get("UserName", None)
            self.id = principal_detail.get("UserId", None)
            policy_list = principal_detail.get("UserPolicyList", None)
            if policy_list:
                logger.debug(f"Adding {self.principal_type}: {self.name}")
                for policy in policy_list:
                    logger.debug(
                        f"\tAdding {policy.get('PolicyName', None)} which is attached to the {self.principal_type} {self.name}"
                    )
                    self.policy_list.append(
                        {
                            "PolicyName": policy.get("PolicyName", None),
                            "PolicyDocument": PolicyDocument(
                                policy.get("PolicyDocument", None)
                            ),
                        }
                    )
        elif self.principal_type == "Group":
            self.name = principal_detail.get("GroupName", None)
            self.id = principal_detail.get("GroupId", None)
            policy_list = principal_detail.get("GroupPolicyList", None)
            if policy_list:
                logger.debug(f"Adding {self.principal_type}: {self.name}")
                for policy in policy_list:
                    logger.debug(
                        f"\tAdding {policy.get('PolicyName', None)} which is attached to the {self.principal_type} {self.name}"
                    )
                    self.policy_list.append(
                        {
                            "PolicyName": policy.get("PolicyName", None),
                            "PolicyDocument": PolicyDocument(
                                policy.get("PolicyDocument", None)
                            ),
                        }
                    )

        elif self.principal_type == "Role":
            self.name = principal_detail.get("RoleName", None)
            self.id = principal_detail.get("RoleId", None)
            policy_list = principal_detail.get("RolePolicyList", None)
            if policy_list:
                logger.debug(f"Adding {self.principal_type}: {self.name}")
                for policy in policy_list:
                    logger.debug(
                        f"\tAdding {policy.get('PolicyName', None)} which is attached to the {self.principal_type} {self.name}"
                    )
                    self.policy_list.append(
                        {
                            "PolicyName": policy.get("PolicyName", None),
                            "PolicyDocument": PolicyDocument(
                                policy.get("PolicyDocument", None)
                            ),
                        }
                    )

    def _principal_policies(self):
        """Stores the list of policies attached to the Principal."""
        principal_policies = []
        if self.principal_type == "User":
            principal_policies = self.principal_detail.get("UserPolicyList", None)
        if self.principal_type == "Group":
            principal_policies = self.principal_detail.get("GroupPolicyList", None)
        if self.principal_type == "Role":
            principal_policies = self.principal_detail.get("RolePolicyList", None)
        return principal_policies

    def _assume_role_policy_document(self):
        """Set the assume role policy document"""
        if self.principal_type == "Role":
            this_document = self.principal_detail.get("AssumeRolePolicyDocument", None)
            if this_document:
                assume_role_policy_document = AssumeRolePolicyDocument(this_document)
                return assume_role_policy_document
            else:
                return None
        else:
            return None

    @property
    def assume_role_from_compute(self):
        """Parse the Trust Policy and determine if an AWS Compute service (EC2, ECS, EKS, Lambda) is able to assume the role."""
        # print(self.assume_role_policy_document)
        if self.principal_type == "Role":
            print()
        else:
            return []
        return self.assume_role_policy_document

    @property
    def account_id(self):
        """Return the account ID"""
        account_id = get_account_from_arn(self.arn)
        return account_id

    def is_principal_excluded(self, exclusions_cfg=DEFAULT_EXCLUSIONS_CONFIG):
        """According to the exclusions configuration, determine whether or not to skip the Principal according
        to their name."""
        decision = False
        if self.principal_type == "User":
            if is_name_excluded(
                get_resource_path_from_arn(self.arn), exclusions_cfg.get("users")
            ):
                decision = True
        if self.principal_type == "Group":
            if is_name_excluded(
                get_resource_path_from_arn(self.arn), exclusions_cfg.get("groups")
            ):
                decision = True
        if self.principal_type == "Role":
            if is_name_excluded(
                get_resource_path_from_arn(self.arn), exclusions_cfg.get("roles")
            ):
                decision = True
        return decision
