"""Represents the entire JSON file generated by the aws iam get-account-authorization-details command."""
# Copyright (c) 2020, salesforce.com, inc.
# All rights reserved.
# Licensed under the BSD 3-Clause license.
# For full license text, see the LICENSE file in the repo root
# or https://opensource.org/licenses/BSD-3-Clause
import logging
from policy_sentry.querying.all import get_all_service_prefixes
from cloudsplaining.shared.constants import DEFAULT_EXCLUSIONS_CONFIG
from cloudsplaining.scan.policy_details import PolicyDetails
from cloudsplaining.scan.principals import PrincipalTypeDetails
from cloudsplaining.output.findings import Findings, Finding
from cloudsplaining.shared.exclusions import is_name_excluded

all_service_prefixes = get_all_service_prefixes()
logger = logging.getLogger(__name__)


class AuthorizationDetails:
    """
    Represents the entire JSON file generated by the aws iam get-account-authorization-details command.
    """

    def __init__(self, auth_json):
        self.policies = PolicyDetails(auth_json.get("Policies", None))
        self.user_detail_list = PrincipalTypeDetails(
            auth_json.get("UserDetailList", None)
        )
        self.group_detail_list = PrincipalTypeDetails(
            auth_json.get("GroupDetailList", None)
        )
        self.role_detail_list = PrincipalTypeDetails(
            auth_json.get("RoleDetailList", None)
        )
        self.findings = Findings()
        self.customer_managed_policies_in_use = self._customer_managed_policies_in_use()
        self.aws_managed_policies_in_use = self._aws_managed_policies_in_use()

    def _aws_managed_policies_in_use(self):
        aws_managed_policies = []
        for policy in self.policies.policy_details:
            if "arn:aws:iam::aws:" in policy.arn:
                aws_managed_policies.append(policy.policy_name)
        # Policies attached to groups
        for principal in self.group_detail_list.principals:
            for attached_managed_policy in principal.attached_managed_policies:
                if "arn:aws:iam::aws:" in attached_managed_policy.get("PolicyArn"):
                    aws_managed_policies.append(
                        attached_managed_policy.get("PolicyName")
                    )
        # Policies attached to users
        for principal in self.user_detail_list.principals:
            for attached_managed_policy in principal.attached_managed_policies:
                if "arn:aws:iam::aws:" in attached_managed_policy.get("PolicyArn"):
                    aws_managed_policies.append(
                        attached_managed_policy.get("PolicyName")
                    )
        # Policies attached to roles
        for principal in self.role_detail_list.principals:
            for attached_managed_policy in principal.attached_managed_policies:
                if "arn:aws:iam::aws:" in attached_managed_policy.get("PolicyArn"):
                    aws_managed_policies.append(
                        attached_managed_policy.get("PolicyName")
                    )

        return aws_managed_policies

    def _customer_managed_policies_in_use(self):
        customer_managed_policies = []
        for policy in self.policies.policy_details:
            if "arn:aws:iam::aws:" not in policy.arn:
                customer_managed_policies.append(policy.policy_name)
        # Policies attached to groups
        for principal in self.group_detail_list.principals:
            for attached_managed_policy in principal.attached_managed_policies:
                if "arn:aws:iam::aws:" not in attached_managed_policy.get("PolicyArn"):
                    customer_managed_policies.append(
                        attached_managed_policy.get("PolicyName")
                    )
        # Policies attached to users
        for principal in self.user_detail_list.principals:
            for attached_managed_policy in principal.attached_managed_policies:
                if "arn:aws:iam::aws:" not in attached_managed_policy.get("PolicyArn"):
                    customer_managed_policies.append(
                        attached_managed_policy.get("PolicyName")
                    )
        # Policies attached to roles
        for principal in self.role_detail_list.principals:
            for attached_managed_policy in principal.attached_managed_policies:
                if "arn:aws:iam::aws:" not in attached_managed_policy.get("PolicyArn"):
                    customer_managed_policies.append(
                        attached_managed_policy.get("PolicyName")
                    )

        return customer_managed_policies

    @property
    def groups(self):
        """A list of the group names in the account, according to the account authorization details."""
        group_names = []
        for principal in self.group_detail_list.principals:
            group_names.append(principal.name)
        return group_names

    @property
    def roles(self):
        """A list of the role names in the account, according to the account authorization details."""
        role_names = []
        for principal in self.role_detail_list.principals:
            role_names.append(principal.name)
        return role_names

    @property
    def users(self):
        """A list of the user names in the account, according to the account authorization details."""
        user_names = []
        for principal in self.user_detail_list.principals:
            user_names.append(principal.name)
        return user_names

    def missing_resource_constraints(
        self, exclusions_cfg=DEFAULT_EXCLUSIONS_CONFIG, modify_only=True
    ):
        """Scan the account authorization details for missing resource constraints."""
        print("-----USERS-----")
        self.scan_principal_type_details(
            self.user_detail_list, exclusions_cfg, modify_only
        )
        print("-----GROUPS-----")
        self.scan_principal_type_details(
            self.group_detail_list, exclusions_cfg, modify_only
        )
        print("-----ROLES-----")
        self.scan_principal_type_details(
            self.role_detail_list, exclusions_cfg, modify_only
        )
        print("-----POLICIES-----")
        self.scan_policy_details(exclusions_cfg, modify_only)
        return self.findings.json

    def scan_policy_details(
        self, exclusions_cfg=DEFAULT_EXCLUSIONS_CONFIG, modify_only=True
    ):
        """Scan the PolicyDetails block of the account authorization details output."""
        excluded_actions = exclusions_cfg.get("exclude-actions", None)

        for policy in self.policies.policy_details:
            print(f"Scanning policy: {policy.policy_name}")
            always_include_actions = exclusions_cfg.get("include-actions")
            actions_missing_resource_constraints = []
            if is_name_excluded(policy.policy_name, exclusions_cfg.get("policies")):
                print(f"\tExcluded policy name: {policy.policy_name}")
            elif is_name_excluded(
                policy.full_policy_path, exclusions_cfg.get("policies")
            ):
                print(f"\tExcluded policy path: {policy.full_policy_path}")
            else:
                for statement in policy.policy_document.statements:
                    if modify_only:
                        if statement.effect == "Allow":
                            actions_missing_resource_constraints.extend(
                                statement.missing_resource_constraints_for_modify_actions(
                                    always_include_actions
                                )
                            )
                    else:
                        if statement.effect == "Allow":
                            actions_missing_resource_constraints.extend(
                                statement.missing_resource_constraints
                            )
                if actions_missing_resource_constraints:
                    actions_missing_resource_constraints = list(
                        dict.fromkeys(actions_missing_resource_constraints)
                    )  # remove duplicates
                    actions_missing_resource_constraints.sort()
                    finding = Finding(
                        policy_name=policy.policy_name,
                        arn=policy.arn,
                        actions=actions_missing_resource_constraints,
                        policy_document=policy.policy_document,
                        always_exclude_actions=excluded_actions
                    )
                    self.findings.add(finding)

    def scan_principal_type_details(
        self,
        principal_type_detail_list,
        exclusions_cfg=DEFAULT_EXCLUSIONS_CONFIG,
        modify_only=True,
    ):
        """Scan the UserDetailList, GroupDetailList, or RoleDetailList blocks of the account authorization details output."""
        excluded_actions = exclusions_cfg.get("exclude-actions", None)

        for principal in principal_type_detail_list.principals:
            always_include_actions = exclusions_cfg.get("include-actions")
            print(f"Scanning {principal.principal_type}: {principal.name}")
            for policy in principal.policy_list:
                print(f"\tScanning Policy: {policy['PolicyName']}")

                if is_name_excluded(
                    policy["PolicyName"], exclusions_cfg.get("policies")
                ):
                    print(f"\tExcluded policy name: {policy['PolicyName']}")
                elif principal.is_principal_excluded(exclusions_cfg):
                    print(f"\tExcluded principal name: {principal.name}")
                else:
                    policy_document = policy["PolicyDocument"]
                    actions_missing_resource_constraints = []
                    for statement in policy_document.statements:
                        if modify_only:
                            if statement.effect == "Allow":
                                actions_missing_resource_constraints.extend(
                                    statement.missing_resource_constraints_for_modify_actions(
                                        always_include_actions
                                    )
                                )
                        else:
                            if statement.effect == "Allow":
                                actions_missing_resource_constraints.extend(
                                    statement.missing_resource_constraints
                                )
                    if actions_missing_resource_constraints:
                        finding = Finding(
                            policy_name=policy["PolicyName"],
                            arn=principal.arn,
                            actions=actions_missing_resource_constraints,
                            policy_document=policy["PolicyDocument"],
                            assume_role_policy_document=principal.assume_role_policy_document,
                            always_exclude_actions=excluded_actions
                        )
                        self.findings.add(finding)
