"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/*! Copyright [Amazon.com](http://amazon.com/), Inc. or its affiliates. All Rights Reserved.
SPDX-License-Identifier: Apache-2.0 */
const crypto = require("crypto");
const aws_sdk_1 = require("aws-sdk"); // eslint-disable-line
const prepare_spec_1 = require("./prepare-spec");
const s3 = new aws_sdk_1.S3();
/**
 * Prepare the api spec for API Gateway
 * @param inputSpecLocation location of the specification to prepare
 * @param outputSpecLocation location to write the prepared spec to
 * @param options integrations, authorizers etc to apply
 * @return the output location of the prepared spec
 */
const prepare = async ({ inputSpecLocation, outputSpecLocation, ...options }) => {
    // Read the spec from the s3 input location
    const inputSpec = JSON.parse((await s3
        .getObject({
        Bucket: inputSpecLocation.bucket,
        Key: inputSpecLocation.key,
    })
        .promise()).Body.toString("utf-8"));
    // Prepare the spec
    const preparedSpec = prepare_spec_1.prepareApiSpec(inputSpec, options);
    const preparedSpecHash = crypto
        .createHash("sha256")
        .update(JSON.stringify(preparedSpec))
        .digest("hex");
    const outputLocation = {
        bucket: outputSpecLocation.bucket,
        key: `${outputSpecLocation.key}/${preparedSpecHash}.json`,
    };
    // Write the spec to the s3 output location
    await s3
        .putObject({
        Bucket: outputLocation.bucket,
        Key: outputLocation.key,
        Body: JSON.stringify(preparedSpec),
    })
        .promise();
    return outputLocation;
};
exports.handler = async (event) => {
    switch (event.RequestType) {
        case "Create":
        case "Update":
            // Prepare the spec on create
            const outputLocation = await prepare(event.ResourceProperties);
            return {
                PhysicalResourceId: outputLocation.key,
                Status: "SUCCESS",
                Data: {
                    outputSpecKey: outputLocation.key,
                },
            };
        case "Delete":
        // Nothing to do for delete
        default:
            break;
    }
    return {
        PhysicalResourceId: event.PhysicalResourceId,
        Status: "SUCCESS",
    };
};
//# sourceMappingURL=data:application/json;base64,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