"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
require("@aws-cdk/assert/jest");
const log = require("npmlog");
const process = require("process");
const override_warning_service_1 = require("../lib/override-warning-service");
const __1 = require("../");
// Mock setup
beforeEach(() => {
    jest.resetModules();
    jest.clearAllMocks();
    delete process.env.overrideWarningsEnabled;
});
// --------------------------------------------------------------
// Test override detection: positive, not-nested
// --------------------------------------------------------------
test('Test override detection: positive, not-nested', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC'
    };
    const b = {
        keyA: 'new-valueA',
        keyD: 'valueD'
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    override_warning_service_1.flagOverriddenDefaults(a, b);
    // Assert
    expect(warn).toBeCalledTimes(1);
});
// --------------------------------------------------------------
// Test override detection: negative, not-nested
// --------------------------------------------------------------
test('Test override detection: negative, not-nested', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC'
    };
    const b = {
        keyD: 'valueD'
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    override_warning_service_1.flagOverriddenDefaults(a, b);
    // Assert
    expect(warn).toBeCalledTimes(0);
});
// --------------------------------------------------------------
// Test override detection: positive, nested
// --------------------------------------------------------------
test('Test override detection: positive, nested', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC',
        keyD: {
            keyDA: 'valueDA',
            keyDB: 'valueDB'
        }
    };
    const b = {
        keyC: 'valueCAnew',
        keyD: {
            keyDA: 'valueDAnew'
        }
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    override_warning_service_1.flagOverriddenDefaults(a, b);
    // Assert
    expect(warn).toBeCalledTimes(2);
});
// --------------------------------------------------------------
// Test override detection: negative, nested
// --------------------------------------------------------------
test('Test override detection: negative, nested', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC',
        keyD: {
            keyDA: 'valueDA',
            keyDB: 'valueDB'
        }
    };
    const b = {
        keyD: {
            keyDA: 'valueDA'
        }
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    override_warning_service_1.flagOverriddenDefaults(a, b);
    // Assert
    expect(warn).toBeCalledTimes(0);
});
// --------------------------------------------------------------
// Test override warning on/off: default on
// --------------------------------------------------------------
test('Test override warning on/off: default on', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC'
    };
    const b = {
        keyA: 'new-valueA',
        keyD: 'valueD'
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    __1.overrideProps(a, b);
    // Assert
    expect(warn).toBeCalledTimes(1);
});
// --------------------------------------------------------------
// Test override warning on/off: explicit on
// --------------------------------------------------------------
test('Test override warning on/off: explicit on', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC'
    };
    const b = {
        keyA: 'new-valueA',
        keyD: 'valueD'
    };
    process.env.overrideWarningsEnabled = 'true';
    // Act
    const warn = jest.spyOn(log, 'warn');
    __1.overrideProps(a, b);
    // Assert
    expect(warn).toBeCalledTimes(1);
});
// --------------------------------------------------------------
// Test override warning on/off: explicit off
// --------------------------------------------------------------
test('Test override warning on/off: explicit off', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC'
    };
    const b = {
        keyA: 'new-valueA',
        keyD: 'valueD'
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    process.env.overrideWarningsEnabled = 'false';
    __1.overrideProps(a, b);
    // Assert
    expect(warn).toBeCalledTimes(0);
});
// --------------------------------------------------------------
// Test override warning on/off: undefined on
// --------------------------------------------------------------
test('Test override warning on/off: undefined on', () => {
    // Arrange
    const a = {
        keyA: 'valueA',
        keyB: 'valueB',
        keyC: 'valueC'
    };
    const b = {
        keyA: 'new-valueA',
        keyD: 'valueD'
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    process.env.overrideWarningsEnabled = undefined;
    __1.overrideProps(a, b);
    // Assert
    expect(warn).toBeCalledTimes(1);
});
// --------------------------------------------------------------
// Test current prefilters
// --------------------------------------------------------------
test('Test current prefilters', () => {
    // Arrange
    const a = {
        destination: 'sampleDestinationValueA',
        maxRecordAge: 'sampleMaxRecordAgeValueA',
        expiration: 'sampleExpirationValueA',
        transitionAfter: 'sampleTransitionValueA',
        timeout: 'sampleTimeoutValueA'
    };
    const b = {
        destination: 'sampleDestinationValueB',
        maxRecordAge: 'sampleMaxRecordAgeValueB',
        expiration: 'sampleExpirationValueB',
        transitionAfter: 'sampleTransitionValueB',
        timeout: 'sampleTimeoutValueB'
    };
    // Act
    const warn = jest.spyOn(log, 'warn');
    process.env.overrideWarningsEnabled = undefined;
    __1.overrideProps(a, b);
    // Assert
    expect(warn).toBeCalledTimes(0);
});
//# sourceMappingURL=data:application/json;base64,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