"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const defaults = require("../index");
require("@aws-cdk/assert/jest");
test('snapshot test buildUserPool default params', () => {
    const stack = new core_1.Stack();
    defaults.buildUserPool(stack);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('snapshot test buildUserPoolClient default params', () => {
    const stack = new core_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    defaults.buildUserPoolClient(stack, userpool);
    expect(assert_1.SynthUtils.toCloudFormation(stack)).toMatchSnapshot();
});
test('Test override for buildUserPool', () => {
    const stack = new core_1.Stack();
    const userpoolProps = {
        userPoolName: 'test',
        signInAliases: { username: false, email: true, phone: true }
    };
    defaults.buildUserPool(stack, userpoolProps);
    expect(stack).toHaveResource('AWS::Cognito::UserPool', {
        UsernameAttributes: [
            "email",
            "phone_number"
        ],
        UserPoolAddOns: {
            AdvancedSecurityMode: "ENFORCED"
        },
        UserPoolName: "test"
    });
});
test('Test override for buildUserPoolClient', () => {
    const stack = new core_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclientProps = {
        userPoolClientName: 'test',
        userPool: userpool
    };
    defaults.buildUserPoolClient(stack, userpool, userpoolclientProps);
    expect(stack).toHaveResource('AWS::Cognito::UserPoolClient', {
        UserPoolId: {
            Ref: "CognitoUserPool53E37E69"
        },
        ClientName: "test"
    });
});
test('Test override for buildIdentityPool', () => {
    const stack = new core_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    defaults.buildIdentityPool(stack, userpool, userpoolclient, {
        allowUnauthenticatedIdentities: true
    });
    expect(stack).toHaveResource('AWS::Cognito::IdentityPool', {
        AllowUnauthenticatedIdentities: true,
        CognitoIdentityProviders: [
            {
                ClientId: {
                    Ref: "CognitoUserPoolClient5AB59AE4"
                },
                ProviderName: {
                    "Fn::GetAtt": [
                        "CognitoUserPool53E37E69",
                        "ProviderName"
                    ]
                },
                ServerSideTokenCheck: true
            }
        ]
    });
});
test('Test setupCognitoForElasticSearch', () => {
    const stack = new core_1.Stack();
    const userpool = defaults.buildUserPool(stack);
    const userpoolclient = defaults.buildUserPoolClient(stack, userpool, {
        userPoolClientName: 'test',
        userPool: userpool
    });
    const identitypool = defaults.buildIdentityPool(stack, userpool, userpoolclient);
    defaults.setupCognitoForElasticSearch(stack, 'test-domain', {
        userpool,
        userpoolclient,
        identitypool
    });
    expect(stack).toHaveResource('AWS::Cognito::UserPoolDomain', {
        Domain: "test-domain"
    });
    expect(stack).toHaveResource('AWS::Cognito::IdentityPoolRoleAttachment', {
        IdentityPoolId: {
            Ref: "CognitoIdentityPool"
        },
        Roles: {
            authenticated: {
                "Fn::GetAtt": [
                    "CognitoAuthorizedRole14E74FE0",
                    "Arn"
                ]
            }
        }
    });
    expect(stack).toHaveResource('AWS::IAM::Role', {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRoleWithWebIdentity",
                    Condition: {
                        "StringEquals": {
                            "cognito-identity.amazonaws.com:aud": {
                                Ref: "CognitoIdentityPool"
                            }
                        },
                        "ForAnyValue:StringLike": {
                            "cognito-identity.amazonaws.com:amr": "authenticated"
                        }
                    },
                    Effect: "Allow",
                    Principal: {
                        Federated: "cognito-identity.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "es:ESHttp*",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":es:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":domain/test-domain/*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "CognitoAccessPolicy"
            }
        ]
    });
});
//# sourceMappingURL=data:application/json;base64,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