"use strict";
/**
 *  Copyright 2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.buildKinesisStreamCWAlarms = exports.buildKinesisStream = void 0;
// Imports
const kinesis = require("@aws-cdk/aws-kinesis");
const kinesis_streams_defaults_1 = require("./kinesis-streams-defaults");
const cdk = require("@aws-cdk/core");
const utils_1 = require("./utils");
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
function buildKinesisStream(scope, props) {
    if (props.existingStreamObj) {
        return props.existingStreamObj;
    }
    // Setup the stream properties
    let kinesisStreamProps;
    if (props.kinesisStreamProps) {
        // If property overrides have been provided, incorporate them and deploy
        kinesisStreamProps = utils_1.overrideProps(kinesis_streams_defaults_1.DefaultStreamProps, props.kinesisStreamProps);
    }
    else {
        // If no property overrides, deploy using the default configuration
        kinesisStreamProps = kinesis_streams_defaults_1.DefaultStreamProps;
    }
    // Create the stream and return
    return new kinesis.Stream(scope, 'KinesisStream', kinesisStreamProps);
}
exports.buildKinesisStream = buildKinesisStream;
function buildKinesisStreamCWAlarms(scope) {
    // Setup CW Alarms for KinesisStream
    const alarms = new Array();
    // Alarm if Max (GetRecords.IteratorAgeMilliseconds): >= 12 hours (50% of 24 hours default retention period)
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamGetRecordsIteratorAgeAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'GetRecords.IteratorAgeMilliseconds'
        }),
        threshold: 2592000,
        evaluationPeriods: 1,
        statistic: 'Maximum',
        period: cdk.Duration.minutes(5),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
        alarmDescription: 'Consumer Record Processing Falling Behind, there is risk for data loss due to record expiration.'
    }));
    // Alarm if Avg (ReadProvisionedThroughputExceeded): > 0
    alarms.push(new cloudwatch.Alarm(scope, 'KinesisStreamReadProvisionedThroughputExceededAlarm', {
        metric: new cloudwatch.Metric({
            namespace: 'AWS/Kinesis',
            metricName: 'ReadProvisionedThroughputExceeded'
        }),
        threshold: 0,
        evaluationPeriods: 1,
        statistic: 'Average',
        period: cdk.Duration.minutes(5),
        comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_THRESHOLD,
        alarmDescription: 'Consumer Application is Reading at a Slower Rate Than Expected.'
    }));
    return alarms;
}
exports.buildKinesisStreamCWAlarms = buildKinesisStreamCWAlarms;
//# sourceMappingURL=data:application/json;base64,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