"""SConsider.site_tools.generateScript.

SConsider-specific stand alone script to execute programs
"""
# vim: set et ai ts=4 sw=4:
# -------------------------------------------------------------------------
# Copyright (c) 2009, Peter Sommerlad and IFS Institute for Software
# at HSR Rapperswil, Switzerland
# All rights reserved.
#
# This library/application is free software; you can redistribute and/or
# modify it under the terms of the license that is included with this
# library/application in the file license.txt.
# -------------------------------------------------------------------------
# pylint: disable=bad-continuation

import os
import datetime
import tempfile
from SCons.Action import Action
from SCons.Builder import Builder
from SConsider.SomeUtils import getPyFilename


def generateShellScript(scriptFile, env, binpath):
    pwdCommand = 'pwd -P'
    defaultExpansions = [
        ('LIBDIR', lambda env: env['LIBDIR']),
        ('BINDIR', lambda env: env['BINDIR']),
        ('CONFIGDIR', lambda env: env['CONFIGDIR']),
        ('SCRIPTDIR', lambda env: env['SCRIPTDIR']),
        ('VARIANTDIR', lambda env: env.getRelativeVariantDirectory()),
        ('BINARYNAME', os.path.basename(binpath)),
        ('BASEDIR', '$(searchBaseDirUp "${SCRIPTPATH}" "${SCRIPTDIR}")'),
    ]

    specificExtensions = []
    if env['PLATFORM'] == 'darwin':
        libpathvariable = 'DYLD_LIBRARY_PATH'
    else:
        libpathvariable = 'LD_LIBRARY_PATH'
    if 'mingw' in env['TOOLS']:
        pwdCommand = 'pwd'
        libpathvariable = 'PATH'
    if 'GENERATESCRIPTVARS' in env:
        specificExtensions += env['GENERATESCRIPTVARS']

    def expandvars(env, expansions=None):
        stringToReturn = ''
        exportvars = 'export'
        if expansions is not None:
            for k, v in expansions:
                try:
                    stringToReturn += k + '="' + (v(env) if callable(v) else str(v)) + '"\n'
                    exportvars += ' ' + k
                except:
                    pass
        if stringToReturn:
            stringToReturn += exportvars + '\n'
        return stringToReturn

    scriptText = """#!/bin/bash
# -------------------------------------------------------------------------
# Copyright (c) """ + datetime.date.today().strftime(
        '%Y') + """, Peter Sommerlad and IFS Institute for Software
# at HSR Rapperswil, Switzerland
# All rights reserved.
#
# This library/application is free software; you can redistribute and/or
# modify it under the terms of the license that is included with this
# library/application in the file license.txt.
# -------------------------------------------------------------------------
# Autogenerated by SConsider; do not edit!

GDBSERVERPORT=2345

MYNAME="$(basename "$0")"
SCRIPTPATH="$(dirname "$0")"
SCRIPTPATH="$(cd "$SCRIPTPATH" 2>/dev/null && """ + pwdCommand + """)"
# shellcheck disable=SC2034
STARTPATH="$(""" + pwdCommand + r""")"

doChangeDir=1
doDebug=0
doDebugServer=0
doTrace=0
declare -a cmdArr

showhelp()
{
    [ -n "${1}" ] && echo "${1}";
    echo ""
    echo "usage: $MYNAME [options]"
    echo "where options are:"
    echo " -d             : run under debugger control (gdb) in foreground"
    echo "                  second -d runs executable in background mode"
    echo "                  and logs stacktrace on failure"
    echo " -e             : run under debugger control in your IDE (gdbserver)"
    echo " -S             : do not change directory before executing target"
    echo "                  eg. stay in current directory for execution"
    echo " -x <tool|opts> : run <tool tool-opts server-command server-options>"
    echo "                  to specify tool options, use multiple -x <opts> options"
    echo " -v             : verbose mode, second -v keeps generated batch files"
    echo .
    exit 3;
}

OPTIND=1
while getopts ":deSvx:-" opt
do
    case $opt in
        :)
            showhelp "ERROR: -$OPTARG parameter missing, exiting!";
        ;;
        d)
            doDebug=$((doDebug + 1));
        ;;
        e)
            doDebugServer=1;
        ;;
        S)
            doChangeDir=0;
        ;;
        v)
            doTrace=$((doTrace + 1));
        ;;
        x)
            cmdArr=("${cmdArr[@]}" "$OPTARG");
        ;;
        -)
            break;
        ;;
        \?)
            showhelp;
        ;;
    esac
done

shift $((OPTIND - 1))

# find base directory for a given path
# param 1: path to start from
# param 2: is the path segment to search for
# param 3: value to return if search failed, default ""
searchBaseDirUp()
{
    start_dir="${1}";
    searchSegment="${2}";
    dirDefault="${3}";
    basePath="$(cd "$start_dir" 2>/dev/null &&
        while [ ! -d "${searchSegment}" ] && [ "$(pwd)" != / ]; do
            cd .. 2>/dev/null;
        done;
        """ + pwdCommand + """
    )";
    test -d "${basePath}/${searchSegment}" || basePath="${dirDefault}";
    echo "$basePath";
}

die() {
    printf "%s\\n" "$@";
    exit 1;
}

""" + expandvars(env, defaultExpansions) + """
# find the base directory
LIBDIR_BASE="$(searchBaseDirUp "${SCRIPTPATH}" "${LIBDIR}")"

test -n "${LIBDIR_BASE}" || die "No base directory found containing library directory [$LIBDIR], exiting."

ABS_LIBDIR="${LIBDIR_BASE}/${LIBDIR}"
test -d "${ABS_LIBDIR}/${VARIANTDIR}" && ABS_LIBDIR="${ABS_LIBDIR}/$VARIANTDIR"

""" + libpathvariable + """="${ABS_LIBDIR}":$""" + libpathvariable + """
export """ + libpathvariable + """

BINDIR_BASE="$(searchBaseDirUp "${SCRIPTPATH}" "${BINDIR}")"

test -n "${BINDIR_BASE}" || die "No base directory found containing binary directory [$BINDIR], exiting."

ABS_BINDIR="${BINDIR_BASE}/${BINDIR}"
test -d "${ABS_BINDIR}/${VARIANTDIR}" && ABS_BINDIR="${ABS_BINDIR}/$VARIANTDIR"

""" + expandvars(env, specificExtensions) + """
# generate gnu debugger command file which may be used for batch
# invocations of the debugger.
#
# param 1: is the name of the generated batch file
# param 2: binary to execute
generateGdbCommandFile()
{
	ggcfBatchFile="${1}";
	ggcfBinaryToExecute="${2}";
	test $# -eq 2 || return 1;
	# <<-EOF ignore tabs, nice for formatting heredocs
cat > "${ggcfBatchFile}" <<-EOF
	handle SIGSTOP nostop nopass
	handle SIGLWP  nostop pass
	handle SIGTERM nostop pass
	handle SIGINT  nostop pass
	set environment PATH=${PATH}
	set environment COAST_ROOT=${COAST_ROOT}
	set environment COAST_PATH=${COAST_PATH}
	set environment """ + libpathvariable + """=${""" + libpathvariable + """}
	set auto-solib-add 1
	# convert to Windows path on mingw (msys supplies it automatically to non-msys tools)
	file \"""" + ("$(cmd //c echo ${ggcfBinaryToExecute})"
               if "mingw" in env["TOOLS"] else "${ggcfBinaryToExecute}") + r'''"
EOF
}

# generate gnu debugger command file which may be used for batch
# invocations of the debugger.
#
# param 1: is the name of the generated batch file
extendGdbCommandFileBatch()
{
	ggcfBatchFile="${1}";
cat >> "$ggcfBatchFile" <<-EOF
	set pagination 0
	run
	if \$_isvoid(\$_siginfo)
			if \$_isvoid(\$_exitcode)
					set \$_exitcode=0
			end
			quit \$_exitcode
	else
			! echo "\$(date +'%Y%m%d%H%M%S'): ========== GDB backtrace =========="
			backtrace full
			info registers
			x/16i \$pc
			thread apply all backtrace
			if !\$_isvoid(\$_siginfo)
					set \$_exitcode=\$_siginfo.si_signo
			end
			if \$_isvoid(\$_exitcode)
					set \$_exitcode=55
			end
			quit \$_exitcode
	end
EOF
}

CMD="${ABS_BINDIR}/${BINARYNAME}"
test -x "${CMD}" || die "binary [${CMD}] is not executable, aborting!"

if [ "${doChangeDir}" -eq 1 ] && [ -d "${BINDIR_BASE}" ]; then
    #shellcheck disable=SC2164
    cd "${BINDIR_BASE}" 2>/dev/null
fi;

test "${doTrace}" -eq 1 && ( cat <<EOF
Executing command [${CMD}]
 with arguments   [$@]
 in directory     [$(pwd)]
EOF
)

toolPath=$(type -fP "${cmdArr[0]}" 2>/dev/null)
if [ -z "$toolPath" ] && [ "${cmdArr[0]}" != "exec" ]; then
  cmdArr=()
fi

fn="$(mktemp)"
fn_gdb="$(mktemp)"
if [ "${doDebugServer:-0}" -ge 1 ]; then
  if [ -x "$(type -fP gdbserver 2>/dev/null)" ]; then
    CMD="gdbserver :${GDBSERVERPORT} ${CMD}"
  else
    echo "gdbserver executable not found, executing cmd only"
  fi
fi
if [ "${doDebug:-0}" -ge 1 ]; then
  if [ -x "$(type -fP gdb 2>/dev/null)" ]; then
    generateGdbCommandFile "$fn_gdb" "$CMD"
    _fn_org="$fn"
    fn="$fn_gdb"
    CMD="set args"
  else
    echo "gdb executable not found, executing cmd only"
    doDebug=0
  fi
fi

_cmd="${cmdArr[*]} ${CMD}"
for i in "$@"; do
  _cmd="$_cmd \"$i\""
done
cat <<-EOF >>"$fn"
$_cmd
EOF

if [ "${doDebug:-0}" -ge 1 ]; then
    cfg_gdbcommands="--command $fn_gdb";
    if [ "$doDebug" -gt 1 ]; then
      cfg_gdbcommands="--batch $cfg_gdbcommands";
      extendGdbCommandFileBatch "$fn_gdb"
    fi
    test "${doTrace}" -ge 1 && {
        echo "Generated gdb commands file for command [$cfg_gdbcommands]:";
        cat "$fn_gdb";
    }
    fn="$_fn_org"
	cat <<-EOF >>"$fn"
		gdb $cfg_gdbcommands
EOF
fi
if [ "${doTrace}" -ge 1 ]; then
	[ "${doTrace}" -lt 2 ] && echo rm -f "$fn" "$fn_gdb" >>"$fn"
	echo "executing [$(cat "$fn")]";
fi
bash "$fn"
'''
    scriptFile.write(scriptText)


def generatePosixScript(target, source, env):
    for t, s in zip(target, source):
        scriptFile = open(str(t), 'w')
        generateShellScript(scriptFile, env, s.get_path())
        scriptFile.close()
    return 0


def generateScriptEmitter(target, source, env):
    target = []
    for src in source:
        script_name = src.name + '.sh'
        if script_name.startswith('bin_'):
            script_name = script_name.partition('_')[2]
        target_basepath = env.getScriptInstallDir().File(script_name)
        if src.has_builder() and src.builder.target_factory:
            target.append(src.builder.target_factory(env.makeInstallablePathFromDir(target_basepath)))
        else:
            target.append(target_basepath)
    return (target, source)


def generateWrapperScript(env, target):
    return env.Depends(env.GenerateScriptBuilder(target), getPyFilename(__file__))


def generate(env):
    from SCons.Tool import install
    from SCons.Defaults import Chmod
    GenerateScriptAction = Action(generatePosixScript, "Creating wrapper script '$TARGET' for '$SOURCE'")
    GenerateScriptBuilder = Builder(action=[GenerateScriptAction,
                                            Chmod('$TARGET', 0755)],
                                    emitter=[generateScriptEmitter, install.add_targets_to_INSTALLED_FILES])

    env.Append(BUILDERS={'GenerateScriptBuilder': GenerateScriptBuilder})
    env.AddMethod(generateWrapperScript, "GenerateWrapperScript")
    env.Append(CONFIGDIR='config')


def exists(env):
    return 1
