from typing import List
from cpanlp.accounting.assets.asset import *
from cpanlp.market.market import *

class LegalEntity:
    """
    #### A legal entity represents a distinct and separate entity created by law, typically for the purpose of conducting business or other activities. Here are some key features of legal entities:

    Features:
    - Name: The name of the legal entity.
    - Type: The legal form or structure of the entity, such as LLC, corporation, partnership, or sole proprietorship.
    - Registration number: The unique identifier assigned to the entity by the government or regulatory authority.
    - Address: The physical location where the entity is registered or operates.
    - Capital: The funds invested in the entity by its owners or shareholders.
    - Employees: The individuals who work for the entity, including managers, staff, and contractors.
    - Assets: The resources owned or controlled by the entity, such as property, equipment, and financial instruments.
    - Partners: The other entities or individuals who have a stake or interest in the entity, such as joint venture partners or investors.
    - Departments: The internal units or divisions within the entity, such as sales, marketing, finance, and operations.
    - Agency cost: The expenses incurred by the entity to mitigate the risk of conflicts of interest between management and shareholders.
    - Market leader: Whether the entity is the leading player in its market or industry.
    - Market share: The percentage of the total market or industry that the entity controls.
    - Sales: The revenue generated by the entity from its operations.
    - Asset: The total value of the entity's assets.
    - Liability: The total amount of the entity's debts and obligations.
    - Investment: The total amount of funds invested by the entity in other entities or assets.
    - Revenue: The total income generated by the entity from all sources.
    - Business scope: The range of products, services, or activities that the entity engages in.
    - Registered capital: The amount of capital registered with the government or regulatory authority.
    - Shareholders: The owners of the entity's stock or equity.
    - Legal representative: The individual authorized to act on behalf of the entity in legal matters.
    - Bankruptcy: Whether the entity is insolvent or unable to meet its financial obligations.
    
    Args:
    - name: The name of the legal entity.
    - type: The legal form or structure of the entity, such as LLC, corporation, partnership, or sole proprietorship.
    - capital: The funds invested in the entity by its owners or shareholders.
    
    Methods:
    - add_department: Adds a department to the entity's internal structure.
    - add_partner: Adds a partner to the entity's ownership structure.
    - fire_employee: Terminates an employee's employment with the entity.
    - hire_employee: Hires a new employee to work for the entity.
    - totalsalary: Calculates the total salary paid to all employees of the entity.
    - merge: Merges the current entity with another entity.
    - spin_off: Creates a new entity as a spin-off of the current entity.
    - increase_capital: Increases the capital of the entity.
    - decrease_capital: Decreases the capital of the entity.
    - enter_market: Enters a new market by creating a market object.
    """
    def __init__(self, name, type=None,capital=None):
        self.name = name
        self.type = type
        self.registration_number=""
        self.address=""
        self.capital=capital
        self.employees =[]
        self.assets=[]
        self.partners = []
        self.departments = []
        self.agency_cost = None
        self.market_leader = False
        self.market_share = None
        self.sales =None
        self.asset = None
        self.liability = 0.0    
        self.investment = 0.0
        self.revenue = None
        self.business_scope = None
        self.registered_capital = 0.0
        self.shareholders = None
        self.legal_representative = None
        self.is_bankrupt = False
    def add_department(self, department):
        self.departments.append(department)
    def add_partner(self, partner):
        self.partners.append(partner)
    def fire_employee(self, employee):
        self.employees.remove(employee)
    def hire_employee(self, employee):
        self.employees.append(employee)
    def totalsalary(self):
        return 0.0 if self.employees is None else sum([member.salary for member in self.employees])
    def merge(self, other_entity):
        """
        Merges the current LLC with another LLC
        """
        # Logic to merge the two LLCs
        self.employees.extend(other_entity.employees)
        self.capital += other_entity.capital
        self.name = f"{self.name}-{other_entity.name}"
    def spin_off(self, spin_off_name:str,spin_off_type:str,spin_off_capital:float):
        """
        Creates a new LLC as a spin-off of the current LLC
        """
        return LegalEntity(spin_off_name,spin_off_type,spin_off_capital)
    def increase_capital(self, amount):
        """
        Increases the capital of the LLC
        """
        self.capital += amount
    def decrease_capital(self, amount):
        """
        Decreases the capital of the LLC
        """
        if self.capital - amount < 0:
            raise ValueError("Capital can not be negative")
        self.capital -= amount
    def enter_market(self, market:Market):
        return market