"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.handler = void 0;
const zlib_1 = require("zlib");
const semver_1 = require("semver");
const aws_lambda_shared_1 = require("../shared/aws.lambda-shared");
const env_lambda_shared_1 = require("../shared/env.lambda-shared");
async function handler({ key, description, widgetContext }) {
    var _a, _b;
    console.log(`Event: ${JSON.stringify({ key, description, widgetContext }, null, 2)}`);
    try {
        const bucketName = env_lambda_shared_1.requireEnv('BUCKET_NAME');
        let { Body, ContentEncoding, LastModified } = await aws_lambda_shared_1.s3().getObject({
            Bucket: bucketName,
            Key: key,
        }).promise();
        // If it was compressed, de-compress it now...
        if (ContentEncoding === 'gzip') {
            Body = zlib_1.gunzipSync(Buffer.from(Body));
        }
        const list = Array.from(JSON.parse(Body.toString('utf-8'))
            .reduce((map, entry) => {
            // Split on the @ that is not at the beginning of the string
            const [name, version] = entry.split(/(?!^)@/);
            if (!map.has(name)) {
                map.set(name, []);
            }
            map.get(name).push(version);
            return map;
        }, new Map())
            .entries())
            .sort(([l], [r]) => l.localeCompare(r));
        // Trying to ensure we don't cause the dashboard to hang due to large DOM.
        const maxCount = 100;
        const objectUrl = `${widgetContext.domain}/s3/object/${bucketName}?prefix=${key}`;
        return {
            markdown: [
                description,
                ...(list.length > maxCount
                    ? [
                        `Showing only the first ${maxCount} packages.`,
                        `The complete list can be obtained [from S3](${objectUrl}).`,
                        '',
                    ]
                    : []),
                'Id | Package Name | Count | Versions',
                '--:|--------------|-------|---------',
                ...list.slice(0, maxCount).map(([name, versions], index) => {
                    versions = semver_1.sort(versions).reverse();
                    return `${index + 1} | \`${name}\` | ${versions.length} | ${versions.map((v) => `[\`${v}\`](${s3ConsoleUrl(bucketName, name, v)})`).join(', ')}`;
                }),
                '',
                `Last updated: \`${(_a = LastModified === null || LastModified === void 0 ? void 0 : LastModified.toISOString()) !== null && _a !== void 0 ? _a : 'N/A'}\``,
            ].join('\n'),
        };
    }
    catch (error) {
        if (error instanceof Error) {
            return {
                markdown: [
                    '**⚠️ An error occurred**',
                    `- **name:** \`${error.name}\``,
                    `- **message:** ${error.message}`,
                    '- **stack:**',
                    '  ```',
                    (_b = error.stack) === null || _b === void 0 ? void 0 : _b.replace(/^/g, '  '),
                    '  ```',
                ].join('\n'),
            };
        }
        ;
        throw error;
    }
}
exports.handler = handler;
function s3ConsoleUrl(bucket, packageName, packageVersion) {
    const encodedPrefix = encodeURIComponent(`data/${packageName}/v${packageVersion}/`);
    return `https://s3.console.aws.amazon.com/s3/buckets/${bucket}?prefix=${encodedPrefix}&showversions=false`;
}
//# sourceMappingURL=data:application/json;base64,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