"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackendDashboard = void 0;
const aws_cloudwatch_1 = require("@aws-cdk/aws-cloudwatch");
const core_1 = require("@aws-cdk/core");
const package_versions_table_widget_1 = require("./backend/inventory/package-versions-table-widget");
const constants_1 = require("./backend/shared/constants");
const language_1 = require("./backend/shared/language");
const deep_link_1 = require("./deep-link");
const metric_utils_1 = require("./metric-utils");
class BackendDashboard extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const reports = [[
                new package_versions_table_widget_1.PackageVersionsTableWidget(this, 'UninstallablePackages', {
                    title: 'Package Versions Report | Uninstallable',
                    description: [
                        "These packages could not be installed. Note that currently they will also appear in the 'missing' documentation reports.",
                        '',
                        "The specific error can be found in the package directory inside a file named 'uninstallable'",
                    ].join('\n'),
                    bucket: props.packageData,
                    key: constants_1.UNINSTALLABLE_PACKAGES_REPORT,
                    height: 6,
                    width: 24,
                }),
            ]];
        for (const language of language_1.DocumentationLanguage.ALL) {
            for (const report of this.perLanguageReports(language, props.packageData)) {
                // put every report in a new line
                reports.push([report]);
            }
        }
        new aws_cloudwatch_1.Dashboard(this, 'Reports', {
            dashboardName: `${props.dashboardName}-reports`,
            periodOverride: aws_cloudwatch_1.PeriodOverride.AUTO,
            start: '-P1W',
            widgets: reports,
        });
        new aws_cloudwatch_1.Dashboard(this, 'Graphs', {
            dashboardName: `${props.dashboardName}-graphs`,
            periodOverride: aws_cloudwatch_1.PeriodOverride.AUTO,
            start: '-P1W',
            widgets: [
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Catalog Overview',
                            '',
                            `[button:primary:Package Data](${deep_link_1.s3ObjectUrl(props.packageData)})`,
                            `[button:Catalog Builder](${deep_link_1.lambdaFunctionUrl(props.orchestration.catalogBuilder.function)})`,
                            `[button:Inventory Canary](${deep_link_1.lambdaFunctionUrl(props.inventory.function)})`,
                            `[button:Search Canary Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.inventory.function)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Size',
                        left: [
                            props.inventory.metricSubmoduleCount({ label: 'Submodules' }),
                            props.inventory.metricPackageVersionCount({ label: 'Package Versions' }),
                            props.inventory.metricPackageMajorCount({ label: 'Package Majors' }),
                            props.inventory.metricPackageCount({ label: 'Packages' }),
                        ],
                        leftYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Catalog Issues',
                        left: [
                            props.inventory.metricUnknownObjectCount({ label: 'Unknown' }),
                            props.inventory.metricMissingAssemblyCount({ label: 'Missing Assembly' }),
                            props.inventory.metricMissingPackageMetadataCount({ label: 'Missing Metadata' }),
                            props.inventory.metricMissingPackageTarballCount({ label: 'Missing Tarball' }),
                            props.inventory.metricUninstallablePackageCount({ label: 'Uninstallable Package' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.catalogBuilder.metricMissingConstructFrameworkCount({ label: 'No Construct Framework' }),
                            props.orchestration.catalogBuilder.metricMissingConstructFrameworkVersionCount({ label: 'No Construct Framework Version' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                ],
                ...this.catalogOverviewLanguageSections(props),
                ...renderPackageSourcesWidgets(props.packageSources),
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Ingestion Function',
                            '',
                            `[button:Ingestion Function](${deep_link_1.lambdaFunctionUrl(props.ingestion.function)})`,
                            `[button:primary:Search Log Group](${deep_link_1.lambdaSearchLogGroupUrl(props.ingestion.function)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.ingestion.deadLetterQueue)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Function Health',
                        left: [
                            metric_utils_1.fillMetric(props.ingestion.function.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(props.ingestion.function.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Queue',
                        left: [
                            props.ingestion.queue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages', period: core_1.Duration.minutes(1) }),
                            props.ingestion.queue.metricApproximateNumberOfMessagesNotVisible({ label: 'Hidden Messages', period: core_1.Duration.minutes(1) }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.queue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age', period: core_1.Duration.minutes(1) }),
                        ],
                        rightAnnotations: [{
                                color: '#ffa500',
                                label: '10 Minutes',
                                value: core_1.Duration.minutes(10).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Input Quality',
                        left: [
                            metric_utils_1.fillMetric(props.ingestion.metricInvalidAssembly({ label: 'Invalid Assemblies' })),
                            metric_utils_1.fillMetric(props.ingestion.metricInvalidTarball({ label: 'Invalid Tarball' })),
                            metric_utils_1.fillMetric(props.ingestion.metricIneligibleLicense({ label: 'Ineligible License' })),
                            metric_utils_1.fillMetric(props.ingestion.metricMismatchedIdentityRejections({ label: 'Mismatched Identity' })),
                            metric_utils_1.fillMetric(props.ingestion.metricFoundLicenseFile({ label: 'Found License file' })),
                        ],
                        leftYAxis: { label: 'Count', min: 0, showUnits: false },
                        stacked: true,
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letters',
                        left: [
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.ingestion.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.ingestion.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightAnnotations: [{
                                color: '#ff7f0e',
                                label: '10 days',
                                value: core_1.Duration.days(10).toSeconds(),
                            }, {
                                color: '#ff0000',
                                label: '14 days (DLQ Retention)',
                                value: core_1.Duration.days(14).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Orchestration',
                            '',
                            `[button:primary:State Machine](${deep_link_1.stateMachineUrl(props.orchestration.stateMachine)})`,
                            `[button:DLQ](${deep_link_1.sqsQueueUrl(props.orchestration.deadLetterQueue)})`,
                            `[button:Redrive DLQ](${deep_link_1.lambdaFunctionUrl(props.orchestration.redriveFunction)})`,
                            `[button:Regenerate All Documentation](${deep_link_1.stateMachineUrl(props.orchestration.regenerateAllDocumentation)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'State Machine Executions',
                        left: [
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricStarted({ label: 'Started' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricSucceeded({ label: 'Succeeded' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricAborted({ label: 'Aborted' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricFailed({ label: 'Failed' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricThrottled({ label: 'Throttled' })),
                            metric_utils_1.fillMetric(props.orchestration.stateMachine.metricTimedOut({ label: 'Timed Out' })),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.stateMachine.metricTime({ label: 'Duration' }),
                        ],
                        rightYAxis: { min: 0 },
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Dead Letter Queue',
                        left: [
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesVisible({ label: 'Visible Messages' }),
                            props.orchestration.deadLetterQueue.metricApproximateNumberOfMessagesNotVisible({ label: 'Invisible Messages' }),
                        ],
                        leftYAxis: { min: 0 },
                        right: [
                            props.orchestration.deadLetterQueue.metricApproximateAgeOfOldestMessage({ label: 'Oldest Message Age' }),
                        ],
                        rightAnnotations: [{
                                color: '#ff7f0e',
                                label: '10 days',
                                value: core_1.Duration.days(10).toSeconds(),
                            }, {
                                color: '#ff0000',
                                label: '14 days (DLQ Retention)',
                                value: core_1.Duration.days(14).toSeconds(),
                            }],
                        rightYAxis: { min: 0 },
                        period: core_1.Duration.minutes(1),
                    }),
                ],
                // deny list
                // ----------------------------------------------
                [
                    new aws_cloudwatch_1.TextWidget({
                        height: 2,
                        width: 24,
                        markdown: [
                            '# Deny List',
                            '',
                            `[button:primary:Deny List Object](${deep_link_1.s3ObjectUrl(props.denyList.bucket, props.denyList.objectKey)})`,
                            `[button:Prune Function](${deep_link_1.lambdaFunctionUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Prune Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.pruneHandler)})`,
                            `[button:Delete Queue](${deep_link_1.sqsQueueUrl(props.denyList.prune.queue)})`,
                            `[button:Delete Logs](${deep_link_1.lambdaSearchLogGroupUrl(props.denyList.prune.deleteHandler)})`,
                        ].join('\n'),
                    }),
                ],
                [
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Deny List',
                        left: [
                            metric_utils_1.fillMetric(props.denyList.metricDenyListRules({ label: 'Rules' }), 'REPEAT'),
                            props.denyList.prune.queue.metricNumberOfMessagesDeleted({ label: 'Deleted Files' }),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                    new aws_cloudwatch_1.GraphWidget({
                        height: 6,
                        width: 12,
                        title: 'Prune Function Health',
                        left: [
                            metric_utils_1.fillMetric(props.denyList.prune.pruneHandler.metricInvocations({ label: 'Invocations' })),
                            metric_utils_1.fillMetric(props.denyList.prune.pruneHandler.metricErrors({ label: 'Errors' })),
                        ],
                        leftYAxis: { min: 0 },
                        period: core_1.Duration.minutes(5),
                    }),
                ],
                ...(props.packageStats ? renderPackageStatsWidgets(props.packageStats) : []),
            ],
        });
    }
    perLanguageReports(language, packageData) {
        return [
            new package_versions_table_widget_1.PackageVersionsTableWidget(this, `MissingDocs-${language.name}`, {
                title: `Package Versions Report | Missing Documentation | _${language.name}_`,
                description: [
                    `These packages are missing ${language.name} documentation.`,
                    'Apart from the uninstallable packages, this report should stay empty',
                    '',
                    'To investigate inspect the orchestration DLQ.',
                ].join('\n'),
                bucket: packageData,
                key: constants_1.missingDocumentationReport(language),
                height: 6,
                width: 24,
            }),
            new package_versions_table_widget_1.PackageVersionsTableWidget(this, `CorruptAssemblyDocs-${language.name}`, {
                title: `Package Versions Report | Corrupt Assembly | _${language.name}_`,
                description: [
                    `These packages are missing ${language.name} documentation because of a corrupted assembly.`,
                    '',
                    "The specific error can be found in the package directory inside files suffixed with '.corruptassembly'",
                ].join('\n'),
                bucket: packageData,
                key: constants_1.corruptAssemblyReport(language),
                height: 6,
                width: 24,
            }),
        ];
    }
    *catalogOverviewLanguageSections({ inventory, orchestration }) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    '# Documentation Generation',
                    '',
                    `[button:primary:Transliterator Logs](${deep_link_1.logGroupUrl(orchestration.transliterator.logGroup)})`,
                    `[button:Transliterator ECS Cluster](${deep_link_1.ecsClusterUrl(orchestration.ecsCluster)})`,
                ].join('\n'),
            }),
        ];
        const mFargateUsage = new aws_cloudwatch_1.Metric({
            dimensionsMap: {
                Class: 'None',
                Resource: 'OnDemand',
                Service: 'Fargate',
                Type: 'Resource',
            },
            metricName: 'ResourceCount',
            namespace: 'AWS/Usage',
            statistic: aws_cloudwatch_1.Statistic.MAXIMUM,
        });
        yield [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Fargate Resources',
                left: [
                    mFargateUsage.with({ label: 'Fargate Usage (On-Demand)' }),
                    new aws_cloudwatch_1.MathExpression({
                        expression: 'SERVICE_QUOTA(mFargateUsage)',
                        label: 'Fargate Quota (On-Demand)',
                        usingMetrics: { mFargateUsage },
                    }),
                ],
                leftYAxis: { min: 0 },
                right: [
                    orchestration.metricEcsCpuUtilization({ label: 'CPU Utilization' }),
                    orchestration.metricEcsMemoryUtilization({ label: 'Memory Utilization' }),
                ],
                rightYAxis: { label: 'Percent', min: 0, max: 100, showUnits: false },
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'ECS Resources',
                left: [
                    metric_utils_1.fillMetric(orchestration.metricEcsNetworkRxBytes({ label: 'Received Bytes' })),
                    metric_utils_1.fillMetric(orchestration.metricEcsNetworkTxBytes({ label: 'Transmitted Bytes' })),
                ],
                leftYAxis: { min: 0 },
                right: [
                    metric_utils_1.fillMetric(orchestration.metricEcsTaskCount({ label: 'Task Count' })),
                ],
                rightYAxis: { min: 0 },
            }),
        ];
        for (const language of language_1.DocumentationLanguage.ALL) {
            yield [
                new aws_cloudwatch_1.TextWidget({
                    height: 1,
                    width: 24,
                    markdown: `## Language: ${language.toString()}`,
                }),
            ];
            yield [
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 12,
                    title: 'Package Versions',
                    left: [
                        inventory.metricSupportedPackageVersionCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricCorruptAssemblyPackageVersionCount(language, { label: 'Corrupt Assembly', color: '#3542D7' }),
                        inventory.metricUnsupportedPackageVersionCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingPackageVersionCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
                new aws_cloudwatch_1.GraphWidget({
                    height: 6,
                    width: 12,
                    title: 'Package Version Submodules',
                    left: [
                        inventory.metricSupportedSubmoduleCount(language, { label: 'Available', color: '#2ca02c' }),
                        inventory.metricCorruptAssemblySubmoduleCount(language, { label: 'Corrupt Assembly', color: '#3542D7' }),
                        inventory.metricUnsupportedSubmoduleCount(language, { label: 'Unsupported', color: '#9467bd' }),
                        inventory.metricMissingSubmoduleCount(language, { label: 'Missing', color: '#d62728' }),
                    ],
                    leftYAxis: { showUnits: false },
                    view: aws_cloudwatch_1.GraphWidgetView.PIE,
                }),
            ];
        }
    }
}
exports.BackendDashboard = BackendDashboard;
function* renderPackageSourcesWidgets(packageSources) {
    var _a;
    for (const packageSource of packageSources) {
        yield [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    `# ${packageSource.name}`,
                    '',
                    ...((_a = packageSource.links) !== null && _a !== void 0 ? _a : [])
                        .map(({ name, primary, url }) => `[${primary ? 'button:primary' : 'button'}:${name}](${url})`),
                ].join('\n'),
            }),
        ];
        yield* packageSource.dashboardWidgets;
    }
    return;
}
function renderPackageStatsWidgets(packageStats) {
    return [
        [
            new aws_cloudwatch_1.TextWidget({
                height: 2,
                width: 24,
                markdown: [
                    '# Package Stats',
                    '',
                    `[button:primary:Package Stats Object](${deep_link_1.s3ObjectUrl(packageStats.bucket, packageStats.statsKey)})`,
                    `[button:Package Stats Function](${deep_link_1.lambdaFunctionUrl(packageStats.handler)})`,
                    `[button:Package Stats Logs](${deep_link_1.lambdaSearchLogGroupUrl(packageStats.handler)})`,
                ].join('\n'),
            }),
        ],
        [
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Number of Package Stats Recorded',
                left: [
                    metric_utils_1.fillMetric(packageStats.metricPackagesCount({ label: 'Packages with stats' }), 'REPEAT'),
                ],
                leftYAxis: { min: 0 },
            }),
            new aws_cloudwatch_1.GraphWidget({
                height: 6,
                width: 12,
                title: 'Invocation Duration',
                left: [
                    packageStats.handler.metricDuration({ label: 'Duration' }),
                ],
                leftYAxis: { min: 0 },
                rightAnnotations: [{
                        color: '#ffa500',
                        label: '15 minutes (Lambda timeout)',
                        value: core_1.Duration.minutes(15).toSeconds(),
                    }],
            }),
        ],
    ];
}
//# sourceMappingURL=data:application/json;base64,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