# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['expression',
 'expression.collections',
 'expression.core',
 'expression.effect',
 'expression.extra',
 'expression.extra.option',
 'expression.extra.result',
 'expression.system']

package_data = \
{'': ['*']}

install_requires = \
['typing-extensions>=4.1.1,<5.0.0']

setup_kwargs = {
    'name': 'expression',
    'version': '4.2.2',
    'description': 'Practical functional programming for Python 3.9+',
    'long_description': '# Expression\n\n[![PyPI](https://img.shields.io/pypi/v/expression.svg)](https://pypi.python.org/pypi/Expression)\n![Python package](https://github.com/cognitedata/expression/workflows/Python%20package/badge.svg)\n![Upload Python Package](https://github.com/cognitedata/expression/workflows/Upload%20Python%20Package/badge.svg)\n[![Documentation Status](https://readthedocs.org/projects/expression/badge/?version=latest)](https://expression.readthedocs.io/en/latest/?badge=latest)\n[![codecov](https://codecov.io/gh/cognitedata/expression/branch/main/graph/badge.svg)](https://codecov.io/gh/cognitedata/expression)\n\n> Pragmatic functional programming\n\nExpression aims to be a solid, type-safe, pragmatic, and high performance\nlibrary for frictionless and practical functional programming in Python 3.9+.\n\nBy pragmatic, we mean that the goal of the library is to use simple abstractions\nto enable you to do practical and productive functional programming in Python\n(instead of being a [Monad tutorial](https://github.com/dbrattli/OSlash)).\n\nPython is a multi-paradigm programming language that also supports functional\nprogramming constructs such as functions, higher-order functions, lambdas, and\nin many ways favors composition over inheritance.\n\n> Better Python with F#\n\nExpression tries to make a better Python by providing several functional\nfeatures inspired by [F#](https://fsharp.org). This serves several\npurposes:\n\n- Enable functional programming in a Pythonic way, i.e., make sure we are not\n  over-abstracting things. Expression will not require purely functional\n  programming as would a language like Haskell.\n- Everything you learn with Expression can also be used with F#. Learn F# by\n  starting in a programming language they already know. Perhaps get inspired to\n  also [try out F#](https://aka.ms/fsharphome) by itself.\n- Make it easier for F# developers to use Python when needed, and re-use many\n  of the concepts and abstractions they already know and love.\n\nExpression will enable you to work with Python using many of the same\nprogramming concepts and abstractions. This enables concepts such as [Railway\noriented programming](https://fsharpforfunandprofit.com/rop/) (ROP) for better\nand predictable error handling. Pipelining for workflows, computational\nexpressions, etc.\n\n> _Expressions evaluate to a value. Statements do something._\n\nF# is a functional programming language for .NET that is succinct (concise,\nreadable, and type-safe) and kind of\n[Pythonic](https://docs.python.org/3/glossary.html). F# is in many ways very\nsimilar to Python, but F# can also do a lot of things better than Python:\n\n- Strongly typed, if it compiles it usually works making refactoring much\n  safer. You can trust the type-system. With [mypy](http://mypy-lang.org/) or\n  [Pylance](https://github.com/microsoft/pylance-release) you often wonder who\n  is right and who is wrong.\n- Type inference, the compiler deduces types during compilation\n- Expression based language\n\n## Getting Started\n\nYou can install the latest `expression` from PyPI by running `pip` (or\n`pip3`). Note that `expression` only works for Python 3.9+.\n\n```console\n> pip3 install expression\n```\n\n## Goals\n\n- Industrial strength library for functional programming in Python.\n- The resulting code should look and feel like Python\n  ([PEP-8](https://www.python.org/dev/peps/pep-0008/)). We want to make a\n  better Python, not some obscure DSL or academic Monad tutorial.\n- Provide pipelining and pipe friendly methods. Compose all the things!\n- Dot-chaining on objects as an alternative syntax to pipes.\n- Lower the cognitive load on the programmer by:\n  - Avoid currying, not supported in Python by default and not a well known\n    concept by Python programmers.\n  - Avoid operator (`|`, `>>`, etc) overloading, this usually confuses more\n    than it helps.\n  - Avoid recursion. Recursion is not normally used in Python and any use of it\n    should be hidden within the SDK.\n- Provide [type-hints](https://docs.python.org/3/library/typing.html) for all\n  functions and methods.\n- Support PEP 634 and structural pattern matching.\n- Code must pass strict static type checking by\n  [pylance](https://devblogs.microsoft.com/python/announcing-pylance-fast-feature-rich-language-support-for-python-in-visual-studio-code/).\n  Pylance is awesome, use it!\n- [Pydantic](https://pydantic-docs.helpmanual.io/) friendly data types. Use Expression\n  types as part of your Pydantic data model and (de)serialize to/from JSON.\n\n## Supported features\n\nExpression will never provide you with all the features of F# and .NET. We are\nproviding a few of the features we think are useful, and will add more\non-demand as we go along.\n\n- **Pipelining** - for creating workflows.\n- **Composition** - for composing and creating new operators.\n- **Fluent or Functional** syntax, i.e., dot chain or pipeline operators.\n- **Pattern Matching** - an alternative flow control to `if-elif-else`.\n- **Error Handling** - Several error handling types.\n  - **Option** - for optional stuff and better `None` handling.\n  - **Result** - for better error handling and enables railway-oriented\n    programming in Python.\n  - **Try** - a simpler result type that pins the error to an Exception.\n- **Collections** - immutable collections.\n  - **TypedArray** - a generic array type that abstracts the details of\n    `bytearray`, `array.array` and `list` modules.\n  - **Sequence** - a better\n    [itertools](https://docs.python.org/3/library/itertools.html) and\n    fully compatible with Python iterables.\n  - **Block** - a frozen and immutable list type.\n  - **Map** - a frozen and immutable dictionary type.\n  - **AsyncSeq** - Asynchronous iterables.\n  - **AsyncObservable** - Asynchronous observables. Provided separately\n    by [aioreactive](https://github.com/dbrattli/aioreactive).\n- **Data Modelling** - sum and product types\n  - **TaggedUnion** - A tagged (discriminated) union type.\n- **Parser Combinators** - A recursive decent string parser combinator\n  library.\n- **Effects**: - lightweight computational expressions for Python. This\n  is amazing stuff.\n  - **option** - an optional world for working with optional values.\n  - **result** - an error handling world for working with result values.\n- **Mailbox Processor**: for lock free programming using the [Actor\n  model](https://en.wikipedia.org/wiki/Actor_model).\n- **Cancellation Token**: for cancellation of asynchronous (and\n  synchronous) workflows.\n- **Disposable**: For resource management.\n\n### Pipelining\n\nExpression provides a `pipe` function similar to `|>` in F#. We don\'t want to\noverload any Python operators, e.g., `|` so `pipe` is a plain old function taking\nN-arguments, and will let you pipe a value through any number of functions.\n\n```python\nfrom expression import pipe\n\nv = 1\nfn = lambda x: x + 1\ngn = lambda x: x * 2\n\nassert pipe(v, fn, gn) == gn(fn(v))\n```\n\nExpression objects (e.g., `Some`, `Seq`, `Result`) also have a `pipe` method, so you can dot chain pipelines\ndirectly on the object:\n\n```python\nfrom expression import Some\n\nv = Some(1)\nfn = lambda x: x.map(lambda y: y + 1)\ngn = lambda x: x.map(lambda y: y * 2)\n\nassert v.pipe(fn, gn) == gn(fn(v))\n```\n\nSo for example with sequences you may create sequence transforming\npipelines:\n\n```python\nfrom expression.collections import seq, Seq\n\nxs = Seq.of(9, 10, 11)\nys = xs.pipe(\n    seq.map(lambda x: x * 10),\n    seq.filter(lambda x: x > 100),\n    seq.fold(lambda s, x: s + x, 0)\n)\n\nassert ys == 110\n```\n\n### Composition\n\nFunctions may even be composed directly into custom operators:\n\n```python\nfrom expression import compose\nfrom expression.collections import seq, Seq\n\nxs = Seq.of(9, 10, 11)\ncustom = compose(\n    seq.map(lambda x: x * 10),\n    seq.filter(lambda x: x > 100),\n    seq.fold(lambda s, x: s + x, 0)\n)\nys = custom(xs)\n\nassert ys == 110\n```\n\n### Fluent and Functional\n\nExpression can be used both with a fluent or functional syntax (or both.)\n\n#### Fluent syntax\n\nThe fluent syntax uses methods and is very compact. But it might get you into\ntrouble for large pipelines since it\'s not a natural way of adding line breaks.\n\n```python\nfrom expression.collections import Seq\n\nxs = Seq.of(1, 2, 3)\nys = xs.map(lambda x: x * 100).filter(lambda x: x > 100).fold(lambda s, x: s + x, 0)\n```\n\nNote that fluent syntax is probably the better choice if you use mypy\nfor type checking since mypy may have problems inferring types through\nlarger pipelines.\n\n#### Functional syntax\n\nThe functional syntax is a bit more verbose but you can easily add new\noperations on new lines. The functional syntax is great to use together\nwith pylance/pyright.\n\n```python\nfrom expression import pipe\nfrom expression.collections import seq, Seq\n\nxs = Seq.of(1, 2, 3)\nys = pipe(xs,\n    seq.map(lambda x: x * 100),\n    seq.filter(lambda x: x > 100),\n    seq.fold(lambda s, x: s + x, 0),\n)\n```\n\nBoth fluent and functional syntax may be mixed and even pipe can be used\nfluently.\n\n```python\nfrom expression.collections import seq, Seq\nxs = Seq.of(1, 2, 3).pipe(seq.map(...))\n```\n\n### Option\n\nThe `Option` type is used when a function or method cannot produce a meaningful\noutput for a given input.\n\nAn option value may have a value of a given type, i.e., `Some(value)`, or it might\nnot have any meaningful value, i.e., `Nothing`.\n\n```python\nfrom expression import Some, Nothing, Option\n\ndef keep_positive(a: int) -> Option[int]:\n    if a > 0:\n        return Some(a)\n\n    return Nothing\n```\n\n```python\nfrom expression import Option, Ok\ndef exists(x : Option[int]) -> bool:\n    match x:\n        case Some(_):\n            return True\n    return False\n```\n\n### Option as an effect\n\nEffects in Expression is implemented as specially decorated coroutines\n([enhanced generators](https://www.python.org/dev/peps/pep-0342/)) using\n`yield`, `yield from` and `return` to consume or generate optional values:\n\n```python\nfrom expression import effect, Some\n\n@effect.option[int]()\ndef fn():\n    x = yield 42\n    y = yield from Some(43)\n\n    return x + y\n\nxs = fn()\n```\n\nThis enables ["railway oriented\nprogramming"](https://fsharpforfunandprofit.com/rop/), e.g., if one part of the\nfunction yields from `Nothing` then the function is side-tracked\n(short-circuit) and the following statements will never be executed. The end\nresult of the expression will be `Nothing`. Thus results from such an option\ndecorated function can either be `Ok(value)` or `Error(error_value)`.\n\n```python\nfrom expression import effect, Some, Nothing\n\n@effect.option[int]()\ndef fn():\n    x = yield from Nothing # or a function returning Nothing\n\n    # -- The rest of the function will never be executed --\n    y = yield from Some(43)\n\n    return x + y\n\nxs = fn()\nassert xs is Nothing\n```\n\nFor more information about options:\n\n- [Tutorial](https://expression.readthedocs.io/en/latest/tutorial/optional_values.html)\n- [API reference](https://expression.readthedocs.io/en/latest/reference/option.html)\n\n### Result\n\nThe `Result[T, TError]` type lets you write error-tolerant code that can be\ncomposed. A Result works similar to `Option`, but lets you define the value used\nfor errors, e.g., an exception type or similar. This is great when you want to\nknow why some operation failed (not just `Nothing`). This type serves the same\npurpose of an `Either` type where `Left` is used for the error condition and `Right`\nfor a success value.\n\n```python\nfrom expression import effect, Ok, Result\n\n@effect.result[int, Exception]()\ndef fn():\n    x = yield from Ok(42)\n    y = yield from Ok(10)\n    return x + y\n\nxs = fn()\nassert isinstance(xs, Result)\n```\n\nA simplified type called `Try` is also available. It\'s a result type that is\npinned to `Exception` i.e., `Result[TSource, Exception]`.\n\n### Sequence\n\nSequences is a thin wrapper on top of iterables and contains operations for working with\nPython iterables. Iterables are immutable by design, and perfectly suited for functional\nprogramming.\n\n```python\nimport functools\nfrom expression import pipe\nfrom expression.collections import seq\n\n# Normal python way. Nested functions are hard to read since you need to\n# start reading from the end of the expression.\nxs = range(100)\nys = functools.reduce(lambda s, x: s + x, filter(lambda x: x > 100, map(lambda x: x * 10, xs)), 0)\n\n# With Expression, you pipe the result, so it flows from one operator to the next:\nzs = pipe(\n    xs,\n    seq.map(lambda x: x * 10),\n    seq.filter(lambda x: x > 100),\n    seq.fold(lambda s, x: s + x, 0),\n)\nassert ys == zs\n```\n\n## Tagged Unions\n\nTagged Unions (aka discriminated unions) may look similar to normal Python Unions. But\nthey are [different](https://stackoverflow.com/a/61646841) in that the operands in a\ntype union `(A | B)` are both types, while the cases in a tagged union type `U = A | B`\nare both constructors for the type U and are not types themselves. One consequence is\nthat tagged unions can be nested in a way union types might not.\n\nIn Expression you make a tagged union by defining your type as a sub-class of\n`TaggedUnion` with the appropriate generic types that this union represent for\neach case. Then you define static or class-method constructors for creating each of the\ntagged union cases.\n\n```python\nfrom dataclasses import dataclass\nfrom expression import TaggedUnion, tag\n\n@dataclass\nclass Rectangle:\n    width: float\n    length: float\n\n@dataclass\nclass Circle:\n    radius: float\n\nclass Shape(TaggedUnion):\n    RECTANGLE = tag(Rectangle)\n    CIRCLE = tag(Circle)\n\n    @staticmethod\n    def rectangle(width: float, length: float) -> Shape:\n        return Shape(Shape.RECTANGLE, Rectangle(width, length))\n\n    @staticmethod\n    def circle(radius: float) -> Shape:\n        return Shape(Shape.CIRCLE, Circle(radius))\n```\n\nNow you may pattern match the shape to get back the actual value:\n\n```python\n    from expression import match\n\n    shape = Shape.Rectangle(2.3, 3.3)\n\n    match shape:\n        case Shape(value=Rectangle(width=2.3)):\n            assert shape.value.width == 2.3\n        case _:\n            assert False\n```\n\n## Notable differences between Expression and F#\n\nIn F# modules are capitalized, in Python they are lowercase\n([PEP-8](https://www.python.org/dev/peps/pep-0008/#package-and-module-names)).\nE.g in F# `Option` is both a module (`OptionModule` internally) and a\ntype. In Python the module is `option` and the type is capitalized i.e\n`Option`.\n\nThus in Expression you use `option` as the module to access module functions\nsuch as `option.map` and the name `Option` for the type itself.\n\n```pycon\n>>> from expression import Option, option\n>>> Option\n<class \'expression.core.option.Option\'>\n>>> option\n<module \'expression.core.option\' from \'/Users/dbrattli/Developer/Github/Expression/expression/core/option.py\'>\n```\n\n## Common Gotchas and Pitfalls\n\nA list of common problems and how you may solve it:\n\n### Expression is missing the function/operator I need\n\nRemember that everything is just a function, so you can easily implement\na custom function yourself and use it with Expression. If you think the\nfunction is also usable for others, then please open a PR to include it\nwith Expression.\n\n## Resources and References\n\nA collection of resources that were used as reference and inspiration\nfor creating this library.\n\n- F# (http://fsharp.org)\n- Get Started with F# (https://aka.ms/fsharphome)\n- F# as a Better Python - Phillip Carter - NDC Oslo 2020\n  (https://www.youtube.com/watch?v=_QnbV6CAWXc)\n- OSlash (https://github.com/dbrattli/OSlash)\n- RxPY (https://github.com/ReactiveX/RxPY)\n- PEP 8 -- Style Guide for Python Code (https://www.python.org/dev/peps/pep-0008/)\n- PEP 342 -- Coroutines via Enhanced Generators\n  (https://www.python.org/dev/peps/pep-0342/)\n- PEP 380 -- Syntax for Delegating to a Subgenerator\n  (https://www.python.org/dev/peps/pep-0380)\n- PEP 479 -- Change StopIteration handling inside generators (https://www.python.org/dev/peps/pep-0479/)\n- PEP 634 -- Structural Pattern Matching (https://www.python.org/dev/peps/pep-0634/)\n- Thunks, Trampolines and Continuation Passing\n  (https://jtauber.com/blog/2008/03/30/thunks,_trampolines_and_continuation_passing/)\n- Tail Recursion Elimination\n  (http://neopythonic.blogspot.com/2009/04/tail-recursion-elimination.html)\n- Final Words on Tail Calls\n  (http://neopythonic.blogspot.com/2009/04/final-words-on-tail-calls.html)\n- Python is the Haskell You Never Knew You Had: Tail Call Optimization\n  (https://sagnibak.github.io/blog/python-is-haskell-tail-recursion/)\n\n## How-to Contribute\n\nYou are very welcome to contribute with suggestions or PRs :heart_eyes: It is\nnice if you can try to align the code and naming with F# modules, functions,\nand documentation if possible. But submit a PR even if you should feel unsure.\n\nCode, doc-strings, and comments should also follow the [Google Python Style\nGuide](https://google.github.io/styleguide/pyguide.html).\n\nCode checks are done using\n\n- [Black](https://github.com/psf/black)\n- [flake8](https://github.com/PyCQA/flake8)\n- [isort](https://github.com/PyCQA/isort)\n\nTo run code checks on changed files every time you commit, install the pre-commit hooks\nby running:\n\n```console\n> pre-commit install\n```\n\n## Code of Conduct\n\nThis project follows https://www.contributor-covenant.org, see our [Code\nof\nConduct](https://github.com/cognitedata/Expression/blob/main/CODE_OF_CONDUCT.md).\n\n## License\n\nMIT, see [LICENSE](https://github.com/cognitedata/Expression/blob/main/LICENSE).\n',
    'author': 'Dag Brattli',
    'author_email': 'dag.brattli@cognite.com',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': 'https://github.com/cognitedata/Expression',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.9,<4',
}


setup(**setup_kwargs)
