# -*- coding: utf-8 -*-
"""plotting.ipynb

Automatically generated by Colaboratory.

Original file is located at
    https://colab.research.google.com/drive/1DjgQ3HJZMJZ8XMZbauij0rmP-koSbpa8
"""

import numpy as np
import matplotlib.pyplot as plt
import pickle
from matplotlib.collections import LineCollection

# Goals
# 0) By default ignore 0 [x]
# 1) Normalize width [x]
# 2) draw connected segments as polygons [x]
# 3) sort segments by  time or size [x]


def visData2Array(visualization_data):
  """
  Returns the visualization_data as an array.
  """
  # Find all unique IDs
  unique_IDs = set()
  for frame_ID in visualization_data.keys():
    frame = visualization_data[frame_ID]
    for segment_ID, segment_count in frame:
      unique_IDs.add(segment_ID)

  # Make an empty array with size frames, uniqueIDs
  frame_count = len(visualization_data.keys())
  segment_count = len(unique_IDs)
  data_array = np.zeros((frame_count, segment_count))

  # Fill the data_array with the values per frame per segment
  for frame_ID in range(frame_count):
    frame = visualization_data[frame_ID]
    for segment_ID, segment_count in frame:
      data_array[frame_ID, segment_ID] = segment_count
  return data_array

def normArray(data_array, maximum=1):
  """
  Returns the normalized array and multiplies with the maximum.
  """
  data_array = (data_array / data_array.max())*maximum
  return data_array

def pointilizeSegment(segment_array):
  """
  Returns the x and y coordinates for the polygon
  representing the particle count at each frame.
  """
  # Start with lower left of polygon
  left_x = -segment_array/2
  # Remove all infs and set to zero
  left_x[left_x ==  np.inf] = 0
  left_x[left_x == -np.inf] = 0
  # Double and roll one for sharp transitions
  left_x = np.repeat(left_x, 2)
  left_x = np.roll(left_x, 1) 
  # Move from top left to top right
  right_x = -np.flip(left_x)
  # Merge xs
  all_x = np.concatenate((left_x, right_x))
  # Start with lower left of polygon
  left_y = np.arange(segment_array.shape[0])
  # Double for for sharp transitions
  left_y = np.repeat(left_y, 2)
  # Move from top left to top right
  right_y = np.flip(left_y)
  # Merge ys
  all_y = np.concatenate((left_y, right_y))
  return all_x, all_y

def order_data_array(data_array, ordered_segments):
  """
  Returns the ordered_segments in the data_array.
  """
  data_array = data_array[:, ordered_segments]
  return data_array

def orderSegments(data_array):
  """
  Returns the sorted segmentIDs from high to low particle count.
  """
  # sort the segment numbers based on max particle count
  order_dict = {}
  for i in range(data_array.shape[1]):
    order_dict[i] = max(data_array[:,i])
  sorted_keys = sorted(order_dict, key=order_dict.get)
  sorted_keys.reverse()
  return np.asarray(sorted_keys)

def plotAllSegments(visualization_data, zero=False, sorting=False, 
                    maximum=10, dpi=300):
  """
  Plots all segments. By default segment zero is not plotted. Sorting sorts
  the segments on their size. The maximum indicates the maximum amount of 
  segments plotted.
  """
  # Handling the non-zero plotting paramater
  if not zero:
    data_array = visData2Array(visualization_data)[:,1:]
  elif zero:
    data_array = visData2Array(visualization_data)
  # Handling the sorting parameter
  if sorting:
    ordered_segments = orderSegments(data_array)
    data_array = order_data_array(data_array, ordered_segments)
  # Handling the mamximum plotting parameter
  if maximum:
    data_array = data_array[:, :maximum]
    ordered_segments = ordered_segments[:maximum]
  # Normalizing the array
  norm_array = normArray(data_array) 
  # Instantiating the plotting
  plt.figure(figsize=(8 ,8))
  plt.xlabel('segment IDs')
  plt.ylabel('frames')
  # Plotting individul segments
  for idx in range(data_array.shape[1]):
    xs, ys = pointilizeSegment(norm_array[:,idx])
    xs *= 0.9
    xs += idx
    # Shifting by one if segment zero is skipped
    if not zero:
      xs += 1
    plt.fill(xs, ys)
  # Handling ticks if the sorting is turned on
  if sorting:
    if not zero:
      plt.xticks(range(1,data_array.shape[1]+1), ordered_segments+1)
    elif zero:
      plt.xticks(range(0,data_array.shape[1]+0), ordered_segments)
  # Handling ticks if sorting is turned off and there are not too many
  #  segments to show individual ticks.
  elif not sorting and data_array.shape[1] < 20:
    if not zero:
      plt.xticks(range(1,data_array.shape[1]+1))
    elif zero:
      plt.xticks(range(0,data_array.shape[1]+0))
  # Saving, showing and closing the final figure
  plt.tight_layout()
  plt.savefig('segmentsPlot.png', dpi = dpi)
  plt.savefig('segmentsPlot.svg')
  plt.show()
  plt.close()
  return

def main():
  # Reading in the segmentation data pickle
  with open('visualization_data.pickle', 'rb') as handle:
      visualization_data = pickle.load(handle)
  # Sorting turns on sorting by sizes, maximum is the maximum amount of 
  #  segments plotted. If sorting is turned on the first so many in size
  #  will be ploted. If sorting is off the first (1-N) will be plotted.
  #  By default the 0 cluster is not plotted, but this can be turned on
  #  by setting zero=True. The data will be plotted to svg and png (using
  #  the indicated dpi).
  plotAllSegments(visualization_data, zero=True, sorting=True, maximum=15, dpi=300)


if __name__ == '__main__':
  main()
