import unittest

from roseta import trans


class TestTransform(unittest.TestCase):
    def setUp(self) -> None:
        self.no_cls_data = {
            "零": (0, "no"),
            "零点八": (0.8, "no"),
            "八": (8, "no"),
            "八十": (80, "no"),
            "八十八": (88, "no"),
            "负八十八": (-88, "no"),
            "负零点八厘米": (-0.8, "cm"),
            "零点八厘米": (0.8, "cm"),
            "负零点八米": (-80, "cm"),
            "零点八米": (80, "cm"),
            "负一米八": (-180, "cm"),
            "一米八": (180, "cm"),
            "一米八七": (187, "cm"),
            "一米8": (180, "cm"),
            "一米87": (187, "cm"),
            "1米八": (180, "cm"),
            "1米八七": (187, "cm"),
            "1米8": (180, "cm"),
            "1米87": (187, "cm"),
            "负零点八斤": (-0.4, "kg"),
            "零点八斤": (0.4, "kg"),
            "零点八公斤": (0.8, "kg"),
            "零点八克": (0.0008, "kg"),
            "零点八千克": (0.8, "kg"),
            "八十八g": (0.088, "kg"),
            "八十八kg": (88, "kg"),
        }

        self.length_data = {
            # pattern1
            "二尺": [(66.67, "cm"), (0.6667, "m")],
            "两尺": [(66.67, "cm"), (0.6667, "m")],
            "2尺": [(66.67, "cm"), (0.6667, "m")],
            "2.1尺": [(70, "cm"), (0.7, "m")],
            "2点1尺": [(70, "cm"), (0.7, "m")],
            "一尺八": [(60, "cm"), (0.6, "m")],
            "1尺一五": [(38.33, "cm"), (0.3833, "m")],
            "二尺五": [(83.33, "cm"), (0.8333, "m")],
            "2尺3": [(76.67, "cm"), (0.7667, "m")],
            "2尺四": [(80, "cm"), (0.8, "m")],
            "2尺2寸": [(73.33, "cm"), (0.7333, "m")],
            "2尺多": [(73.33, "cm"), (0.7333, "m")],
            "两尺一寸": [(70, "cm"), (0.7, "m")],
            "两尺几": [(83.33, "cm"), (0.8333, "m")],
            # pattern2
            "负零点八": [(-0.8, "cm"), (-0.8, "m")],
            "负零点八厘米": [(-0.8, "cm"), (-0.008, "m")],
            "零点八米": [(80, "cm"), (0.8, "m")],
            "零点八厘米": [(0.8, "cm"), (0.008, "m")],
            "八米": [(800, "cm"), (8, "m")],
            "八厘米": [(8, "cm"), (0.08, "m")],
            "八十米": [(8000, "cm"), (80, "m")],
            "八十厘米": [(80, "cm"), (0.8, "m")],
            "八十七米": [(8700, "cm"), (87, "m")],
            "八十七厘米": [(87, "cm"), (0.87, "m")],
            # pattern3
            "-0.8米": [(-80, "cm"), (-0.8, "m")],
            "0.8米": [(80, "cm"), (0.8, "m")],
            "0.8厘米": [(0.8, "cm"), (0.008, "m")],
            "1米": [(100, "cm"), (1, "m")],
            "1厘米": [(1, "cm"), (0.01, "m")],
            "1.8米": [(180, "cm"), (1.8, "m")],
            "1.8厘米": [(1.8, "cm"), (0.018, "m")],
            "18米": [(1800, "cm"), (18, "m")],
            "18厘米": [(18, "cm"), (0.18, "m")],
            "180米": [(18000, "cm"), (180, "m")],
            "180厘米": [(180, "cm"), (1.8, "m")],
            # pattern4
            "负一米八": [(-180, "cm"), (-1.8, "m")],
            "一米八": [(180, "cm"), (1.8, "m")],
            "一米八七": [(187, "cm"), (1.87, "m")],
            "一米8": [(180, "cm"), (1.8, "m")],
            "一米87": [(187, "cm"), (1.87, "m")],
            "1米八": [(180, "cm"), (1.8, "m")],
            "1米八七": [(187, "cm"), (1.87, "m")],
            "1米8": [(180, "cm"), (1.8, "m")],
            "1米87": [(187, "cm"), (1.87, "m")],
            # pattern5
            "八十几厘米": [(85, "cm"), (0.85, "m")],
            "八十多厘米": [(82, "cm"), (0.82, "m")],
            "一米八几": [(185, "cm"), (1.85, "m")],
            "一米八多": [(182, "cm"), (1.82, "m")],
            # pattern6
            "80几": [(85, "cm"), (85, "m")],
            "80几米": [(8500, "cm"), (85, "m")],
            "80几厘米": [(85, "cm"), (0.85, "m")],
            "80多": [(82, "cm"), (82, "m")],
            "80多米": [(8200, "cm"), (82, "m")],
            "80多厘米": [(82, "cm"), (0.82, "m")],
        }

        self.weight_data = {
            # pattern1
            "负零点八": [(-0.8, "g"), (-0.8, "kg")],
            "负零点八斤": [(-400, "g"), (-0.4, "kg")],
            "零点八斤": [(400, "g"), (0.4, "kg")],
            "零点八公斤": [(800, "g"), (0.8, "kg")],
            "零点八克": [(0.8, "g"), (0.0008, "kg")],
            "零点八千克": [(800, "g"), (0.8, "kg")],
            "八十斤": [(40000, "g"), (40, "kg")],
            "八十公斤": [(80000, "g"), (80, "kg")],
            "八十克": [(80, "g"), (0.08, "kg")],
            "八十千克": [(80000, "g"), (80, "kg")],
            "八十八斤": [(44000, "g"), (44, "kg")],
            "八十八公斤": [(88000, "g"), (88, "kg")],
            "八十八克": [(88, "g"), (0.088, "kg")],
            "八十八千克": [(88000, "g"), (88, "kg")],
            "八十八g": [(88, "g"), (0.088, "kg")],
            "八十八kg": [(88000, "g"), (88, "kg")],
            # pattern2
            "-0.8": [(-0.8, "g"), (-0.8, "kg")],
            "-0.8斤": [(-400, "g"), (-0.4, "kg")],
            "0.8斤": [(400, "g"), (0.4, "kg")],
            "0.8公斤": [(800, "g"), (0.8, "kg")],
            "0.8克": [(0.8, "g"), (0.0008, "kg")],
            "0.8千克": [(800, "g"), (0.8, "kg")],
            "80斤": [(40000, "g"), (40, "kg")],
            "80公斤": [(80000, "g"), (80, "kg")],
            "80克": [(80, "g"), (0.08, "kg")],
            "80千克": [(80000, "g"), (80, "kg")],
            "88斤": [(44000, "g"), (44, "kg")],
            "88公斤": [(88000, "g"), (88, "kg")],
            "88克": [(88, "g"), (0.088, "kg")],
            "88千克": [(88000, "g"), (88, "kg")],
            "88g": [(88, "g"), (0.088, "kg")],
            "88kg": [(88000, "g"), (88, "kg")],
            "88G": [(88, "g"), (0.088, "kg")],
            "88KG": [(88000, "g"), (88, "kg")],
            "88Kg": [(88000, "g"), (88, "kg")],
            "88kG": [(88000, "g"), (88, "kg")],
            # # pattern3
            "负一斤八": [(-900, "g"), (-0.9, "kg")],
            "一斤八": [(900, "g"), (0.9, "kg")],
            "一斤8": [(900, "g"), (0.9, "kg")],
            "1斤八": [(900, "g"), (0.9, "kg")],
            "1斤8": [(900, "g"), (0.9, "kg")],
            # pattern4
            "八十几斤": [(42500, "g"), (42.5, "kg")],
            "八十多斤": [(41000, "g"), (41, "kg")],
            "八十几公斤": [(85000, "g"), (85, "kg")],
            "八十多公斤": [(82000, "g"), (82, "kg")],
            # pattern5
            "80几": [(85, "g"), (85, "kg")],
            "80多": [(82, "g"), (82, "kg")],
            "80几斤": [(42500, "g"), (42.5, "kg")],
            "80多斤": [(41000, "g"), (41, "kg")],
            "80几公斤": [(85000, "g"), (85, "kg")],
            "80多公斤": [(82000, "g"), (82, "kg")],
            "80几kg": [(85000, "g"), (85, "kg")],
            "80多kg": [(82000, "g"), (82, "kg")],
            "80几g": [(85, "g"), (0.085, "kg")],
            "80多g": [(82, "g"), (0.082, "kg")],
        }

        self.city_data = {
            # pattern1
            "杭州": [("杭州市", "市"), ("浙江省杭州市", "省")],
            "杭州市": [("杭州市", "市"), ("浙江省杭州市", "省")],
            "阿拉善": [("阿拉善盟", "市"), ("内蒙古自治区阿拉善盟", "省")],
            # pattern2
            "浙江省": [("浙江省", "市"), ("浙江省", "省")],
            "浙江": [("浙江省", "市"), ("浙江省", "省")],
            "天津": [("天津市", "市"), ("天津市", "省")],
            "天津市": [("天津市", "市"), ("天津市", "省")],
            "广西": [("广西壮族自治区", "市"), ("广西壮族自治区", "省")],
            "广西壮族自治区": [("广西壮族自治区", "市"), ("广西壮族自治区", "省")],
        }

        self.age_data = {
            "一岁": [(1, "year"), (12, "month")],
            "二个月": [(0.2, "year"), (2, "month")],
            "18岁": [(18, "year"), (216, "month")],
            "一岁半": [(1.5, "year"), (18, "month")],
            "1岁8个月": [(1.7, "year"), (20, "month")],
            "一岁八个月": [(1.7, "year"), (20, "month")],
            "一周岁": [(1, "year"), (12, "month")],
            "两岁零3个月": [(2.2, "year"), (27, "month")],
            "7个月": [(0.6, "year"), (7, "month")],
            "9月": [(0.8, "year"), (9, "month")],
            "13个月": [(1.1, "year"), (13, "month")],
            "10个多月": [(0.8, "year"), (10.1, "month")],
            "2岁多": [(2.2, "year"), (26, "month")],
            "10個月": [(0.8, "year"), (10, "month")]
        }

        self.cup_data = {
            "32a": [([82, 70], "cm"), ([0.82, 0.7], "m")],
            "c36": [([96, 80], "cm"), ([0.96, 0.8], "m")],
            "40c": [([106, 90], "cm"), ([1.06, 0.9], "m")],
            "c42": [([111, 95], "cm"), ([1.11, 0.95], "m")],
            "E48": [([130, 110], "cm"), ([1.3, 1.1], "m")],
            "85B": [([99, 85], "cm"), ([0.99, 0.85], "m")],
            "95F": [([117, 95], "cm"), ([1.17, 0.95], "m")],
            "c80": [([96, 80], "cm"), ([0.96, 0.8], "m")],
            "H110": [([136, 110], "cm"), ([1.36, 1.1], "m")],
            "105g": [([129, 105], "cm"), ([1.29, 1.05], "m")],
            "三十二a": [([82, 70], "cm"), ([0.82, 0.7], "m")],
            "c三十六": [([96, 80], "cm"), ([0.96, 0.8], "m")],
            "四十c": [([106, 90], "cm"), ([1.06, 0.9], "m")],
            "c四十二": [([111, 95], "cm"), ([1.11, 0.95], "m")],
            "E四十八": [([130, 110], "cm"), ([1.3, 1.1], "m")],
            "八十五B": [([99, 85], "cm"), ([0.99, 0.85], "m")],
            "九十五F": [([117, 95], "cm"), ([1.17, 0.95], "m")],
            "c八十": [([96, 80], "cm"), ([0.96, 0.8], "m")],
            "H一百一十": [([136, 110], "cm"), ([1.36, 1.1], "m")],
            "一百零五g": [([129, 105], "cm"), ([1.29, 1.05], "m")],
            "三二a": [([82, 70], "cm"), ([0.82, 0.7], "m")],
            "c三六": [([96, 80], "cm"), ([0.96, 0.8], "m")],
            "四零c": [([106, 90], "cm"), ([1.06, 0.9], "m")],
            "c四二": [([111, 95], "cm"), ([1.11, 0.95], "m")],
            "E四八": [([130, 110], "cm"), ([1.3, 1.1], "m")],
            "八五B": [([99, 85], "cm"), ([0.99, 0.85], "m")],
            "九五F": [([117, 95], "cm"), ([1.17, 0.95], "m")],
            "c八零": [([96, 80], "cm"), ([0.96, 0.8], "m")],
            "H一一零": [([136, 110], "cm"), ([1.36, 1.1], "m")],
            "一零五g": [([129, 105], "cm"), ([1.29, 1.05], "m")]
        }

    def test_trans(self) -> None:
        for key, value in self.no_cls_data.items():
            self.assertEqual(trans(key), value)

        for key, value in self.length_data.items():
            self.assertEqual(trans(key, cls="length", unit="cm"), value[0])
            self.assertEqual(trans(key, cls="length"), value[0])  # unit default: cm
            self.assertEqual(trans(key, unit="cm"), value[0])
            self.assertEqual(trans(key, cls="length", unit="m"), value[1])
            self.assertEqual(trans(key, unit="m"), value[1])

        for key, value in self.weight_data.items():
            self.assertEqual(trans(key, cls="weight", unit="g"), value[0])
            self.assertEqual(trans(key, unit="g"), value[0])
            self.assertEqual(trans(key, cls="weight", unit="kg"), value[1])
            self.assertEqual(trans(key, cls="weight"), value[1])  # unit default: kg
            self.assertEqual(trans(key, unit="kg"), value[1])

        for key, value in self.city_data.items():
            self.assertEqual(trans(key, cls="city", unit="市"), value[0])
            self.assertEqual(trans(key, cls="city"), value[0])  # unit default: 市
            self.assertEqual(trans(key, cls="city", unit="省"), value[1])

        for key, value in self.age_data.items():
            self.assertEqual(trans(key), value[0])
            self.assertEqual(trans(key, cls="age"),  value[0])  # unit default: year
            self.assertEqual(trans(key, cls="age", unit="year"), value[0])
            self.assertEqual(trans(key, cls="age", unit="month"),  value[1])

        # 罩杯不支持无分类自动识别
        for key, value in self.cup_data.items():
            self.assertEqual(trans(key, cls="cup"), value[0])  # default unit: cm
            self.assertEqual(trans(key, cls="cup", unit="cm"), value[0])
            self.assertEqual(trans(key, cls="cup", unit="m"), value[1])


if __name__ == "__main__":
    unittest.main()
