/* eslint-disable import/prefer-default-export */
import debounce from 'lodash.debounce'

import addListener from '../../components/MetaRefresh/add-listener'

const config = {
	minimizableClass: 'player-minimize',
	minimizableSectionSelector: '.sticky-section',
	closeBtnSelector: '.close-sticky-btn',
	scrollDebounceMS: 150,
	playerAmountBeforeMinimize: 0.8,
}

class PlayerMinimizable {
	constructor(playerInstance) {
		this.playerInstance = playerInstance
	}

	get minimizableSection() {
		return this.playerInstance.getContainer().closest(config.minimizableSectionSelector)
	}

	get viewportTrigger() {
		return this.minimizableSection.parentElement
	}

	get container() {
		return this.playerInstance.getContainer()
	}
}

const triggerStickyEvent = (playerInstance, { sticky, closed }) => {
	playerInstance.trigger('allerStickyInteraction', { sticky, closed })
}

const isElementInViewport = el => {
	if (!process.browser) {
		return false
	}

	const parentEl = el.getBoundingClientRect()
	const windowHeight = window.innerHeight || document.documentElement.clientHeight

	return (
		parentEl.bottom > parentEl.height * config.playerAmountBeforeMinimize &&
		parentEl.top < windowHeight * config.playerAmountBeforeMinimize
	)
}

const addListeners = minimizable => {
	if (!process.browser) {
		return
	}

	const { playerInstance, minimizableSection } = minimizable

	const closePlayerEl = document.querySelector(config.closeBtnSelector)

	let isStickyClosed = false

	closePlayerEl.addEventListener('click', () => {
		minimizableSection.classList.remove(config.minimizableClass)
		isStickyClosed = true
		playerInstance.resize()
		triggerStickyEvent(playerInstance, { sticky: false, closed: true })
		playerInstance.pause()
	})

	const debounced = debounce(
		() => resizePlayer(minimizable, isStickyClosed),
		config.scrollDebounceMS,
	)
	addListener('scroll', debounced)

	playerInstance.on('ready', () => resizePlayer(minimizable, isStickyClosed))
}

function resizePlayer(minimizable, isStickyClosed) {
	if (isStickyClosed) return

	const { playerInstance, minimizableSection, viewportTrigger } = minimizable
	const { utils } = playerInstance
	if (playerInstance.getState() === 'idle') {
		return
	}

	const inViewPort = isElementInViewport(viewportTrigger)
	utils.toggleClass(minimizableSection, config.minimizableClass, !inViewPort)
	playerInstance.resize()
	triggerStickyEvent(playerInstance, { sticky: !inViewPort })
}

function initStickyness(requestedId, alreadyStickyPlayer) {
	try {
		if (!alreadyStickyPlayer) {
			return
		}

		const minimizable = new PlayerMinimizable(alreadyStickyPlayer)

		if (alreadyStickyPlayer.id !== requestedId) {
			return
		}
		addListeners(minimizable)
	} catch (e) {
		// eslint-disable-next-line no-console
		console.error('Error while initializing stickiness')
		// eslint-disable-next-line no-console
		console.error(e)
	}
}

export function setupFirstStickyPlayerOnPage() {
	const mediaid = document.querySelector('[data-videoid][data-sticky="true"]').dataset.videoid

	const playerInstance = window.jwplayer(mediaid)
	initStickyness(mediaid, playerInstance)
}
