import { DbAdConfig } from './configs/dagbladet'
import { SeherAdConfig } from './configs/seher'

function build(config, mediaId) {
	const audienceData = getAudienceData('apr_tdc')
	const pageSection = getPageSection()

	return {
		adUnit: {
			bids: [
				{
					bidder: 'adform',
					params: {
						mid: config.adform.mid,
						mkv: `placement:preroll,publisher:${
							config.publisher
						},section:${pageSection},${stringifyAudienceAdform(audienceData)}`,
					},
				},
				{
					bidder: 'appnexus',
					params: {
						invCode: config.appnexus.nonskip.invCode,
						member: '9742',
						video: {
							skippable: false,
							playback_method: ['auto_play_sound_on'],
						},
						keywords: {
							section: pageSection,
							...audienceData,
						},
					},
				},
				{
					bidder: 'appnexus',
					params: {
						invCode: config.appnexus.skip.invCode,
						member: '9742',
						video: {
							skippable: true,
							playback_method: ['auto_play_sound_on'],
						},
						keywords: {
							section: pageSection,
							...audienceData,
						},
					},
				},
			],
			code: `video-ad-${mediaId}`,
			mediaTypes: {
				video: {
					context: 'instream',
				},
			},
		},
		path: config.path,
	}
}

/**
 * Return any saved Audience Project data from localStorage
 * @param {string} key - localStorage key name
 * @return {object}
 */
function getAudienceData(key) {
	key = key ? key : 'apr_tdc'
	try {
		const json = window.localStorage.getItem(key)
		if (json) {
			const cache = JSON.parse(json)
			if (cache.exp > Date.now() && Object.keys(cache.d).length > 0) {
				return cache.d
			}
		}
		return {}
	} catch (err) {
		// eslint-disable-next-line no-console
		console.error('Error in getAudienceData()', err)
		return {}
	}
}

/**
 * Parse enumerate object and return adform compat key-values string
 * @see {@link https://publishers.adform.com/documentation/technical-info/audience-targeting-setup/key-value-pair-targeting-setup/}
 * @param {obj} obj - Enumerate object
 * @return {string}
 */
function stringifyAudienceAdform(obj) {
	const mkv = []
	Object.keys(obj).forEach(key => {
		if (Array.isArray(obj[key])) {
			obj[key].forEach(item => {
				mkv.push(`${key}:${item}`)
			})
		} else {
			mkv.push(`${key}:${obj[key]}`)
		}
	})

	return mkv.join(',')
}

/**
 * @function
 * @param host - hostname of a current website
 * @param mediaId - id of a current video
 * @param autoplay - if video is autoplayable of not, ad depends on that
 * @returns config ready for consuming by prebid request maker
 */
export default (host, mediaId, autoplay) => {
	let preConfig

	switch (host) {
		case 'www.dagbladet.no':
			preConfig = new DbAdConfig(autoplay)
			break
		case 'www.seher.no':
			preConfig = new SeherAdConfig(autoplay)
			break
		default:
			preConfig = new DbAdConfig(autoplay)
	}

	return build(preConfig.getConfig(), mediaId)
}

function getPageSection() {
	const SECTION_SELECTOR = 'meta[property="page:site_section"]'
	const defaultSection = 'video'
	let sectionToReturn
	const sectionMetatag = document.head.querySelector(SECTION_SELECTOR)

	if (sectionMetatag) {
		sectionToReturn = sectionMetatag.getAttribute('content')
	}

	return sectionToReturn || defaultSection
}
