import { ImageSource } from '@aller/shared/types/images'

import { Column } from '../components/FlexCol/types'

export interface DeviceConfig {
	width: number
	quality: number
	media?: number
	height?: number
}

interface Config {
	desktop: DeviceConfig
	tablet: DeviceConfig
	mobile: DeviceConfig
}

const deviceConfig: Config = {
	desktop: {
		width: 980,
		quality: 85,
		media: 75,
	},
	tablet: {
		width: 980,
		quality: 85,
		media: 48,
	},
	mobile: {
		width: 480,
		quality: 70,
	},
}

const convertColumnToFloat = (column?: number) => (column ? column / 12 : 1)

export function getAdaptedConfig(column: Column, ratio: number): Config {
	const config = JSON.parse(JSON.stringify(deviceConfig))

	if (column.xs) {
		config.mobile.width = config.mobile.width * convertColumnToFloat(column.xs)
		config.tablet.width = config.tablet.width * convertColumnToFloat(column.xs)
		config.desktop.width = config.desktop.width * convertColumnToFloat(column.xs)
	}
	if (column.md) {
		config.tablet.width = config.tablet.width * convertColumnToFloat(column.md)
		config.desktop.width = config.desktop.width * convertColumnToFloat(column.md)
	}
	if (column.lg) {
		config.desktop.width = config.desktop.width * convertColumnToFloat(column.lg)
	}

	config.mobile.height = config.mobile.width * ratio
	config.tablet.height = config.tablet.width * ratio
	config.desktop.height = config.desktop.width * ratio

	return config
}

function makeImageUrl(baseUrl: string, configElement: DeviceConfig): string {
	const { width, height, quality } = configElement

	return `${baseUrl}&width=${width}&height=${height}&compression=${quality}`
}

export function generateLabradorImageUrls(
	baseUrl: string,
	ratio: number = 0.5,
	column: Column,
	progressiveJpeg: boolean = false,
): ImageSource[] {
	if (!baseUrl) return []

	const progressivePrefixBase = `/app/bildr/api/v1/process?processors=to-nextgen&image-url=`
	const progressivePrefix =
		process.env.NODE_ENV === 'production'
			? progressivePrefixBase
			: `https://www.kk.no${progressivePrefixBase}`

	const config = getAdaptedConfig(column, ratio)

	const desktop: ImageSource = {
		url: makeImageUrl(baseUrl, config.desktop),
		media: `(min-width: ${config.desktop.media}em)`,
	}

	const tablet: ImageSource = {
		url: makeImageUrl(baseUrl, config.tablet),
		media: `(min-width: ${config.tablet.media}em)`,
	}

	const mobile: ImageSource = {
		url: makeImageUrl(baseUrl, config.mobile),
		media: '',
	}

	if (progressiveJpeg) {
		desktop.url = `${progressivePrefix}${encodeURIComponent(desktop.url)}`
		tablet.url = `${progressivePrefix}${encodeURIComponent(tablet.url)}`
		mobile.url = `${progressivePrefix}${encodeURIComponent(mobile.url)}`
	}

	if (baseUrl.includes('jwplayer')) {
		desktop.url = baseUrl
		return [desktop]
	}

	return [desktop, tablet, mobile]
}
