import { ImageSource } from '../types/images'
import { getAdaptedConfig, DeviceConfig } from './generate-labrador-image-urls'
import { Column } from '../components/FlexCol/types'

interface ImageSize {
	width: number
	height: number
	quality: number
}

const sizes: ImageSize[] = [
	{ width: 100, height: 110, quality: 100 }, // primarily LoftPlugImage
	{ width: 100, height: 150, quality: 100 }, // primarily LoftPlugImage
	{ width: 480, height: 300, quality: 85 }, // primarily Mobile
	{ width: 400, height: 400, quality: 85 },
	{ width: 400, height: 600, quality: 85 },
	{ width: 700, height: 400, quality: 85 },
	{ width: 980, height: 600, quality: 85 },
	{ width: 1000, height: 1000, quality: 85 },
]

function getBestImageSize(width: number, height: number = 100): ImageSize {
	for (const { width: w, height: h, quality } of sizes) {
		if (width <= w && height <= h) {
			return { width: w, height: h, quality }
		}
	}
	return sizes[sizes.length - 1]
}

export function cloudify(
	imgUrl: string,
	{ width, height = 0, quality }: DeviceConfig,
	baseTransformation: string = 'c_fill,g_face',
): string {
	const domain = process.env.NODE_ENV !== 'production' ? 'https://sol.no' : ''

	const standardizedSizes = getBestImageSize(width, height)
	width = standardizedSizes.width
	height = standardizedSizes.height
	quality = standardizedSizes.quality

	let transformation = `${baseTransformation},w_${Math.round(width)}`
	if (height) {
		transformation += `,h_${Math.round(height)}`
	}
	transformation += `,q_${Math.round(quality)}`

	return `${domain}/app/cimg/${transformation}/${encodeURIComponent(imgUrl)}`
}

const generateCloudifyImageUrls = (
	imageUrl: string,
	ratio: number = 0.5,
	column: Column,
): ImageSource[] => {
	if (!imageUrl) return []

	const config = getAdaptedConfig(column, ratio)

	const desktop: ImageSource = {
		url: cloudify(imageUrl, config.desktop),
		media: `(min-width: ${config.desktop.media}em)`,
	}

	const tablet: ImageSource = {
		url: cloudify(imageUrl, config.tablet),
		media: `(min-width: ${config.tablet.media}em)`,
	}

	const mobile: ImageSource = {
		url: cloudify(imageUrl, config.mobile),
		media: '',
	}

	return [desktop, tablet, mobile]
}

export default generateCloudifyImageUrls
