import firebase from 'firebase/app'
import 'firebase/messaging'
import fetch from 'isomorphic-unfetch'
import getConfig from 'next/config'

const { publicRuntimeConfig } = getConfig()

const firebaseConf = {
	apiKey: publicRuntimeConfig.solApiKey,
	authDomain: publicRuntimeConfig.solAuthDomain,
	databaseURL: publicRuntimeConfig.solDatabaseURL,
	projectId: publicRuntimeConfig.solProjectId,
	storageBucket: publicRuntimeConfig.solStorageBucket,
	messagingSenderId: publicRuntimeConfig.solMessagingSenderId,
	publicToken: publicRuntimeConfig.solPublicToken,
}

export default class Messages {
	constructor(config) {
		this.FCMProxy = publicRuntimeConfig.solFcmProxy
		this.FCMSendUrl = 'https://fcm.googleapis.com/fcm/send'

		if (this.configExists()) {
			// Prevent error if someone uses multiple push modules
			if (!firebase.apps.length) {
				if (firebase.messaging.isSupported()) {
					firebase.initializeApp(firebaseConf)
					this.firebase = firebase.messaging()
					this.firebase.usePublicVapidKey(firebaseConf.publicToken)
				}
			}

			if (config.onMessage) {
				this.firebase.onMessage(config.onMessage)
			}
		}
	}

	setToken() {
		if (this.configExists() && firebase.messaging.isSupported()) {
			return this.firebase
				.requestPermission()
				.then(() => {
					return this.firebase.getToken()
				})
				.then(currentToken => {
					// Subscribe device to topic
					return fetch(`${this.FCMProxy}/api/v1/fcm/push/${currentToken}`, {
						method: 'POST',
					}).catch(() => false)
				})
				.catch(err => {})
		}
		return null
	}

	deleteToken() {
		if (this.configExists() && firebase.messaging.isSupported()) {
			return this.firebase.getToken().then(token => {
				token && this.firebase.deleteToken(token)
			})
		}
		return null
	}

	configExists = () => {
		return Object.keys(publicRuntimeConfig).length !== 0 && firebaseConf.messagingSenderId
	}

	static isSupported() {
		return firebase.messaging.isSupported()
	}
}
