import React, { createContext, useContext, useEffect, useState } from 'react'
import PropTypes from 'prop-types'
import fetch from 'isomorphic-unfetch'
import fetchXavierCookieValue from '@aller/shared/utils/fetch-xavier-cookie-value' // eslint-disable-line import/no-extraneous-dependencies
import { useIsMobile } from '@aller/shared/hooks/useIsMobile'
import standardizeArticlesFromHarvester from '../utils/standardize-articles/from-harvester'
import standardizeArticlesFromXavier from '../utils/standardize-articles/from-xavier'

export const XavierMostReadContext = createContext({})
export const XavierContentAdContext = createContext({})
export const PartnerPlugContext = createContext({})

const XavierProvider = ({ children }) => {
	const [mostread, updateMostread] = useState([])
	const [contentads, updateContentads] = useState([])
	const [partnerPlugs, updatePartnerPlugs] = useState([])
	const isMobile = useIsMobile()

	async function fetchContentAds() {
		const cookie = fetchXavierCookieValue()
		const response = await fetch(
			`https://sol.no/app/xavier-cerebro-api/api/v1/www.sol.no/users/${
				cookie ? cookie : ''
			}?pipeline=contentMarketing`,
		).then(async data => data.json())
		if (window && window.blinkLabrador) {
			window.blinkLabrador.trackArticleImpressions(
				'article.preview,article.preview-blink,.article-list>li',
			)
		}

		updateContentads(response.response.items || [])
	}

	async function fetchPartnerPlug() {
		const cookie = fetchXavierCookieValue()
		const response = await fetch(
			`https://sol.no/app/xavier-cerebro-api/api/v1/sol.no/users/${
				cookie ? cookie : ''
			}?pipeline=offers&platform=${isMobile ? 'mobile' : 'desktop'}`,
		).then(data => data.json())

		const xavierResponse = (response && response.response && response.response.items) || []
		const filteredArticles = []
		xavierResponse.forEach(article => {
			if (article.harvesterId === 'FakeTest') {
				filteredArticles.push(article)
			}
		})

		if (filteredArticles.length > 0) {
			const harvesterResponse = await fetch(
				'https://harvester.sol.no/get?q=paywall:%221%22&includeHosts=elbil24.no&limit=4&responseFilter=title,url,paywall,posted,fields.image',
			).then(async data => data.json())
			const articles = standardizeArticlesFromHarvester(
				(harvesterResponse && harvesterResponse.items) || [],
			)
			updatePartnerPlugs(articles)
		}
	}

	async function fetchMostRead() {
		const response = await fetch(
			'https://sol.no/app/xavier-cerebro-api/api/v1/www.sol.no/users/?pipeline=performanceData&sort=clicks',
		).then(async data => data.json())

		const filteredResponse = []
		response.response.items.forEach(item => {
			if (filteredResponse.length < 6 && item.title && item.url) {
				filteredResponse.push(item)
			}
		})

		updateMostread(filteredResponse)
	}

	useEffect(() => {
		fetchMostRead()
		fetchContentAds()
		fetchPartnerPlug()
	}, []) // eslint-disable-line

	return (
		<XavierMostReadContext.Provider value={mostread}>
			<XavierContentAdContext.Provider value={contentads}>
				<PartnerPlugContext.Provider value={partnerPlugs}>{children}</PartnerPlugContext.Provider>
			</XavierContentAdContext.Provider>
		</XavierMostReadContext.Provider>
	)
}
XavierProvider.propTypes = {
	children: PropTypes.node,
}
XavierProvider.defaultProps = {
	children: null,
}

export const useMostRead = () => useContext(XavierMostReadContext)
export const usePartnerPlugs = () => useContext(PartnerPlugContext)

export const useContentAd = num => {
	const contentads = useContext(XavierContentAdContext)
	if (!contentads || (contentads && contentads.length === 0)) {
		return []
	}

	const articlesToReturn = contentads[num]
	return articlesToReturn
}

export default XavierProvider
