/* eslint-disable import/prefer-default-export */
import React from 'react'
import propTypes from 'prop-types'
import styled from 'styled-components'

import { getVariable } from '@aller/shiny/src/utils/get-variable'
import { FontIcon } from '@aller/shiny/src/atoms/FontIcon'

import mathRound from '../../../utils/math-round'

const StyledWeatherButton = styled.div`
	flex: 1 0 100%;
	transition: opacity 0.2s ease-in-out;
	&:hover {
		cursor: pointer;
		opacity: 0.8;
	}
`

const Img = styled.img`
	margin: 0 calc(${getVariable('horizontalBase')} / 3) 0 calc(${getVariable('horizontalBase')} / 3);
	vertical-align: middle;
`

const ExpandIcon = styled(FontIcon)`
	margin: -0.3rem 0 0 0.4rem;
	vertical-align: middle;
	display: inline-block;
`

const ExpandIconOnSmallDisplays = styled(ExpandIcon)`
	display: inline-block;

	@media screen and (min-width: ${getVariable('largeWidth')}) {
		display: none;
	}
`

const ExpandIconOnLargeDisplays = styled(ExpandIcon)`
	display: none;

	@media screen and (min-width: ${getVariable('largeWidth')}) {
		display: inline-block;
	}
`

const Location = styled.span`
	display: none;

	@media screen and (min-width: ${getVariable('largeWidth')}) {
		display: inline;
	}
`

const WeatherButton = ({ weather, upDownArrow, handleDropdown }) => {
	if (weather.length === 0) {
		return null
	}

	const { location, forecast } = weather[0]

	if (!forecast || !location) {
		return null
	}

	return (
		<StyledWeatherButton onClick={() => handleDropdown()}>
			<Location>{location.place}</Location>
			<ExpandIconOnLargeDisplays name={`arrow-alt-${upDownArrow}`} />
			<Img
				alt={forecast.tabular[0].symbol.name}
				src={
					'https://sol.no/s/img/h_30,w_30/static/weather_symbol/' +
					`${forecast.tabular[0].symbol.numberEx}.png`
				}
			/>
			<span>
				{mathRound(forecast.tabular[0].temperature.value)}
				°C
			</span>
			<ExpandIconOnSmallDisplays name={`arrow-alt-${upDownArrow}`} />
		</StyledWeatherButton>
	)
}
WeatherButton.propTypes = {
	weather: propTypes.arrayOf(
		propTypes.shape({
			location: propTypes.shape({
				place: propTypes.string.isRequired,
			}),
			forecast: propTypes.shape({
				tabular: propTypes.arrayOf(
					propTypes.shape({
						symbol: propTypes.shape({
							name: propTypes.string.isRequired,
							numberEx: propTypes.string.isRequired,
						}),
						temperature: propTypes.shape({
							value: propTypes.string.isRequired,
						}),
					}),
				),
			}),
		}),
	).isRequired,
	upDownArrow: propTypes.oneOf(['up', 'down']),
	handleDropdown: propTypes.func,
}
WeatherButton.defaultProps = {
	handleDropdown: () => null,
	upDownArrow: 'down',
}

export { WeatherButton }
