import Cookies from 'js-cookie'
import propTypes from 'prop-types'
import React from 'react'
import styled from 'styled-components'

import { getColor } from '@aller/shiny/src/utils/get-color'
import { getVariable } from '@aller/shiny/src/utils/get-variable'

import weatherApiRequest from '../../../services/weather-api-request'

import { WeatherButton } from './WeatherButton'
import { WeatherDropdown } from './WeatherDropdown'

const StyledWeather = styled.div`
	display: flex;
	flex-wrap: wrap;
	font-family: ${getVariable('mainFont')};
	font-size: ${getVariable('uiRegularSize')};
	font-weight: ${({ fontWeight }) => fontWeight || getVariable('semiBoldWeight')};
	justify-content: flex-end;
	color: ${getColor('white')};
	line-height: 2.4rem;
	padding: calc(${getVariable('verticalBase')} / 2) ${getVariable('horizontalBase')};
	position: static;
	border-radius: 4px;
	background: ${({ background }) => background || 'transparent'};
	text-transform: ${({ textTransform }) => textTransform || 'none'};
	@media screen and (min-width: ${getVariable('largeWidth')}) {
		position: relative;
	}
`

class Weather extends React.Component {
	constructor() {
		super()

		this.handleDropdown = this.handleDropdown.bind(this) // this so 'this' works from child

		this.state = {
			weather: [],
			fromCookie: false,
			displayDropdown: false,
			upDownArrow: 'down',
		}
	}

	componentDidMount() {
		const { preventUpdate } = this.props

		if (preventUpdate) {
			return
		}

		const { settings: { geoLocation: { location = {} } = {} } = {} } = this.context
		const { latitude, longitude } = location || {} // location sometimes gets the value: null
		const weatherCookie = Cookies.get('weather-location')

		// Use geolocation from IP if available.
		let query =
			latitude && longitude
				? `weather?lat=${latitude}&lon=${longitude}&now=true&byHour=true`
				: 'weather?locationId=norge/oslo/oslo/oslo&now=true&byHour=true'

		// Use location from cookie if available.
		if (weatherCookie) {
			const data = JSON.parse(weatherCookie)
			query = `weather?locationId=${data.id}&now=true&byHour=true`
		}

		this.fetchInitialData(query, weatherCookie)
	}

	shouldComponentUpdate(nextProps, nextState) {
		const { weather, fromCookie, displayDropdown } = this.state

		if (nextProps.preventUpdate) {
			return false
		}

		const hasWeatherChanged = weather.length < nextState.weather.length
		const hasCookieChanged = fromCookie !== nextState.fromCookie
		const hasDropdownChanged = displayDropdown !== nextState.displayDropdown

		return hasWeatherChanged || hasCookieChanged || hasDropdownChanged
	}

	fetchInitialData(query, weatherCookie) {
		weatherApiRequest(query).then(data => {
			const weather = data ? [data] : []
			this.setState({
				weather,
				fromCookie: weatherCookie,
			})
		})
	}

	handleDropdown() {
		const { weather, displayDropdown, upDownArrow } = this.state
		if (weather.length === 0) {
			return
		}

		// Need to fetch dropdown data.
		if (weather.length === 1 && weather[0].location) {
			const { municipality } = weather[0].location
			const query = getCorrectQueries(municipality)

			weatherApiRequest(query).then(data => {
				const currentData = weather[0]

				data.unshift(currentData)

				this.setState({
					weather: data,
					displayDropdown: !displayDropdown,
					upDownArrow: upDownArrow === 'down' ? 'up' : 'down',
				})
			})
		} else {
			this.setState({
				displayDropdown: !displayDropdown,
				upDownArrow: upDownArrow === 'down' ? 'up' : 'down',
			})
		}
	}

	render() {
		const { weather, upDownArrow, displayDropdown } = this.state

		if (!weather) {
			return null
		}

		return (
			<StyledWeather {...this.props}>
				<WeatherButton
					weather={weather}
					upDownArrow={upDownArrow}
					handleDropdown={this.handleDropdown}
				/>
				{displayDropdown ? <WeatherDropdown weather={weather} /> : null}
			</StyledWeather>
		)
	}
}
Weather.propTypes = {
	preventUpdate: propTypes.bool,
}
Weather.defaultProps = {
	preventUpdate: false,
}
Weather.contextTypes = {
	settings: propTypes.shape({}),
}

/* eslint-disable max-len */
function getCorrectQueries(municipality) {
	switch (municipality) {
		case 'Oslo':
			return 'weather/bulk?locationIds=norge/sør-trøndelag/trondheim/trondheim,norge/hordaland/bergen/bergen,norge/rogaland/stavanger/stavanger&now=true'
		case 'Bergen':
			return 'weather/bulk?locationIds=norge/oslo/oslo/oslo,norge/sør-trøndelag/trondheim/trondheim,norge/rogaland/stavanger/stavanger&now=true'
		case 'Trondheim':
			return 'weather/bulk?locationIds=norge/oslo/oslo/oslo,norge/hordaland/bergen/bergen,norge/rogaland/stavanger/stavanger&now=true'
		case 'Stavanger':
			return 'weather/bulk?locationIds=norge/oslo/oslo/oslo,norge/hordaland/bergen/bergen,norge/sør-trøndelag/trondheim/trondheim&now=true'
		default:
			return 'weather/bulk?locationIds=norge/oslo/oslo/oslo,norge/hordaland/bergen/bergen,norge/sør-trøndelag/trondheim/trondheim&now=true'
	}
}
/* eslint-enable max-len */

export default Weather
