import React from 'react'
import PropTypes from 'prop-types'
import styled, { css } from 'styled-components'
import dynamic from 'next/dynamic'

import { getVariable } from '@aller/shiny/src/utils/get-variable'
import { FontIcon } from '@aller/shiny/src/atoms/FontIcon'
import { LinkBarItem } from '@aller/shiny/src/atoms/LinkBarItem'
import { LinkBarButton } from '@aller/shiny/src/atoms/LinkBarButton'
import { SmallLinkBarLink, LinkBarLink, LargeLinkBarLink } from '@aller/shiny/src/atoms/LinkBarLink'
import { LargeLinkBarDropdown } from '@aller/shiny/src/atoms/LinkBarDropdown'
import { VerticalLinkBar } from '@aller/shiny/src/molecules/VerticalLinkBar'
import { HorizontalLinkBar } from '@aller/shiny/src/molecules/HorizontalLinkBar'
import { LargeLinkBarSearchField } from '@aller/shiny/src/atoms/LinkBarSearchField'
import UserDropDown from './UserDropDown'

const SeHerLogo = dynamic(() =>
	import(/* webpackChunkName: "SeHerLogo" */ '@aller/shiny/src/atoms/SeHerLogo').then(
		c => c.SeHerLogo,
	),
)

const VanillaDropdown = dynamic(() =>
	import(/* webpackChunkName: "VanillaDropdown" */ '../Vanilla/VanillaDropdown'),
)

const logoWidth = '5.9rem'
const smLogoWidth = '7.2rem'

const LinkBarBleedingLogo = styled(LinkBarItem)`
	align-self: flex-start;
	width: ${logoWidth};
	@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.sm}em) {
		width: ${smLogoWidth};
	}
`

const SeHerDropdownButton = props => {
	const { linkText, hide, ...rest } = props
	const icon = hide === true ? 'list' : 'close'

	return (
		<LinkBarButton {...rest}>
			{linkText} <FontIcon name={icon} />
		</LinkBarButton>
	)
}
SeHerDropdownButton.propTypes = {
	linkText: PropTypes.string,
	hide: PropTypes.bool,
}
SeHerDropdownButton.defaultProps = {
	linkText: '',
	hide: false,
}

const SecondaryBar = styled(HorizontalLinkBar)`
	${props => {
		const horizontalBase = getVariable('horizontalBase')(props)
		const smBreakpoint = props.theme.flexboxgrid.breakpoints.sm

		return css`
			margin-left: calc(${logoWidth} + ${horizontalBase});
			width: calc(100% - ${logoWidth} - ${horizontalBase});

			@media screen and (min-width: ${smBreakpoint}em) {
				margin-left: calc(${smLogoWidth} + ${horizontalBase});
				width: calc(100% - ${smLogoWidth} - ${horizontalBase});
			}
		`
	}};
`

const TopAlignBar = styled(HorizontalLinkBar)`
	flex: 1 0 auto;
	align-items: flex-start;
`

const LogoLink = styled(LargeLinkBarLink)`
	position: absolute;
	top: 0;
	transition: 0.2s width;
	width: ${logoWidth};

	@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.sm}em) {
		width: ${smLogoWidth};
	}
`

const GrowingListItem = styled(LinkBarItem)`
	flex: 1 0 auto;
`

const LinkBarLinkDynamicBold = styled(LargeLinkBarLink)`
	&& {
		@media screen and (max-width: ${({ theme }) => theme.flexboxgrid.breakpoints.sm}rem) {
			padding-right: 0.7rem;
		}
	}
`

const linkProps = {
	activeBackgroundColor: 'transparent',
	textColor: 'type',
	skin: 'splashBorder',
	ALLCAPS: true,
	useUnderline: false,
}

const secondaryLinkProps = { ...linkProps, textColor: 'typeLight' }

const MobileMenu = styled(LargeLinkBarDropdown)`
	margin-right: calc(-1 / 2 * ${getVariable('horizontalBase')});
`

// Give the dropdown menu some padding where it overlaps with the seher logo
const StyledVerticalLinkBar = styled(VerticalLinkBar)`
	padding-top: calc(3 / 4 * ${getVariable('verticalBase')});
	@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.sm}em) {
		padding-top: 0;
	}
`

const SeHerHeader = ({ displayDropdownsInitially, elementType, ...rest }) => (
	<header {...rest}>
		<TopAlignBar
			backgroundColor="white"
			isTopLevelComponent={false} // Use the full width
			shouldFlexChildren
			justifyContent="space-between"
			overflow="visible"
			zIndex={200}
		>
			<GrowingListItem isListItem>
				<TopAlignBar
					backgroundColor="transparent"
					isTopLevelComponent={false}
					overflow="visible"
					shouldFlexChildren
				>
					<LinkBarBleedingLogo isListItem>
						<LogoLink {...linkProps} useUnderline={false} isBlockLink url="/" title="Se&Hør">
							<SeHerLogo />
						</LogoLink>
					</LinkBarBleedingLogo>

					<LargeLinkBarLink xs={false} md url="/kongelig" {...linkProps}>
						Kongelig
					</LargeLinkBarLink>
					<LargeLinkBarLink xs={false} md url="/brudd-bryllup" {...linkProps}>
						Brudd &amp; bryllup
					</LargeLinkBarLink>
					<LargeLinkBarLink xs={false} md url="/video" {...linkProps}>
						Video
					</LargeLinkBarLink>

					<VanillaDropdown
						Dropdown={LargeLinkBarDropdown}
						id="commercial-dropdown"
						xs={false}
						lg
						linkText="Kommersielt"
						{...linkProps}
					>
						<VerticalLinkBar backgroundColor="white">
							<LinkBarLink linkText="Reise" url="/reise" {...linkProps} />
							<LinkBarLink
								linkText="Rabattkoder"
								url="https://www.kk.no/rabattkoder"
								{...linkProps}
							/>
						</VerticalLinkBar>
					</VanillaDropdown>
				</TopAlignBar>
			</GrowingListItem>
			<LinkBarLinkDynamicBold
				id="seher-login-button"
				{...linkProps}
				linkText="Logg Inn"
				url="/app/minside"
			/>
			<UserDropDown selector="seher" />
			<LargeLinkBarSearchField
				action="https://www.seher.no/tekstarkiv/"
				placeholder="Søk i Se&Hør..."
				xs={false}
				md
				backgroundColor="grayTintLighter"
				iconColor="typeLight"
				icon="search"
			/>
			<VanillaDropdown
				Dropdown={MobileMenu}
				id="xs-menu"
				displayInitially={displayDropdownsInitially}
				{...linkProps}
				xs
				md={false}
				position="static"
				linkText="Meny"
				textColor="type"
				Trigger={SeHerDropdownButton}
				zIndex={7}
			>
				<StyledVerticalLinkBar backgroundColor="white" align="right">
					<LinkBarLink url="/kongelig" {...linkProps}>
						Kongelig
					</LinkBarLink>
					<LinkBarLink url="/brudd-bryllup" {...linkProps}>
						Brudd &amp; bryllup
					</LinkBarLink>
					<LinkBarLink url="/video" {...linkProps}>
						Video
					</LinkBarLink>
				</StyledVerticalLinkBar>
			</VanillaDropdown>
		</TopAlignBar>
		<SecondaryBar
			isTopLevelComponent={false} // Use the full width
			shouldFlexChildren
			justifyContent="flex-start"
			background="linear-gradient(
				180deg,
				rgba(229,229,229,0.50),
				#f3f3f3 1.3rem
			)"
		>
			<SmallLinkBarLink linkText="Rød løper" url="/video" {...secondaryLinkProps} />
			<SmallLinkBarLink linkText="Reality" url="/emne/reality" {...secondaryLinkProps} />
			<SmallLinkBarLink linkText="Brudd & bryllup" url="/brudd-bryllup" {...secondaryLinkProps} />
			<SmallLinkBarLink linkText="Hollywood" url="/emne/hollywood" {...secondaryLinkProps} />
			<SmallLinkBarLink linkText="Kongelig" url="/kongelig" {...secondaryLinkProps} />
			<SmallLinkBarLink linkText="Video" url="/video" {...secondaryLinkProps} />
			<SmallLinkBarLink
				linkText="Sterke historier"
				url="/sterke-historier"
				{...secondaryLinkProps}
			/>
		</SecondaryBar>
	</header>
)

// We are spreading props in render-children, which makes React warn us thinking we might want some fields
//  as DOM attributes and that we not use e.g. `elementType` but `element-type` instead.
// These prop types have been added to remove props that produce warnings when spreading rest:
// - elementType
SeHerHeader.propTypes = {
	elementType: PropTypes.string,
	displayDropdownsInitially: PropTypes.bool,
	user: PropTypes.shape({ name: PropTypes.string, id: PropTypes.string }),
}
SeHerHeader.defaultProps = {
	elementType: '',
	displayDropdownsInitially: false,
	user: undefined,
}

export default SeHerHeader
