import React, { useState } from 'react'
import dynamic from 'next/dynamic'
import styled from 'styled-components'
import { getColor, getVariable } from '@aller/shiny/src/utils'
import { LinkBarLink } from '@aller/shiny/src/atoms/LinkBarLink'
import { LargeLinkBarSearchField } from '@aller/shiny/src/atoms/LinkBarSearchField'
import { LinkBarItem } from '@aller/shiny/src/atoms/LinkBarItem'
import { HorizontalLinkBar } from '@aller/shiny/src/molecules/HorizontalLinkBar'
import { Col } from '@aller/shiny/src/atoms/Col'
import { SvgIcon } from '@aller/shiny/src/atoms/SvgIcon'
import UserDropDown from './DagbladetHeader/UserDropDown'
import { IconButton } from './DagbladetHeader/IconButton'

const LinkBarProps = {
	useUnderline: false,
	isActive: false,
	activeBackground: 'primary',
	size: 'small',
}

const linkProps = {
	useUnderline: false,
	activeBackgroundColor: 'primaryDark',
	textColor: 'white',
	activeTextColor: 'white',
	ALLCAPS: true,
}

const mobileMenuLinkProps = {
	useUnderline: false,
	activeBackgroundColor: 'primary',
	textColor: 'white',
	activeTextColor: 'white',
	ALLCAPS: true,
}

const mobileGetBackMenuLinkProps = {
	useUnderline: false,
	activeBackgroundColor: '#383843',
	textColor: 'white',
	activeTextColor: 'white',
}

const TopAlignHeaderProps = {
	backgroundColor: getColor('primaryDark'),
	isTopLevelComponent: false,
	shouldFlexChildren: true,
	justifyContent: 'space-between',
	overflow: 'visible',
	zIndex: 200,
}

const Header = styled.header`
	padding: 0;
`

const GetBack = styled.span`
	font-weight: 400;
	font-size: calc(${getVariable('headingRegularSize')} / 2);
	opacity: 0.8;
	@media screen and (min-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		font-size: calc(${getVariable('headingLargeSize')} - 1rem);
	}
`

const CategoryLink = styled(LinkBarLink)`
	line-height: 0;
	padding: 0;
`

const StyledLinkBarLink = styled(LinkBarLink)`
	letter-spacing: 0.1rem;
	transition: 0.2s;
	line-height: 3rem;
	padding: 1rem 2rem;
	font-weight: 600;
	font-size: ${getVariable('uiSmallSize')};
	&&:hover {
		transition: 0.2s;
	}
`

const LinkBarLinkDynamicBold = styled(LinkBarLink)`
	right: 9rem;
	@media screen and (min-width: ${({ theme }) => theme.flexboxgrid.breakpoints.sm}rem) {
		right: 20rem;
		font-weight: ${getVariable('uiWeightBold')};
	}
`

const UserDropDownWrapper = styled(UserDropDown)`
	line-height: 0;
`

const SearchFieldDynamicWidth = styled(LargeLinkBarSearchField)`
	&& {
		height: 3rem;
		width: 15rem;
		background-color: ${getColor('darkness')};
		border-radius: 0.4rem;
		border: 0.1rem solid ${getColor('darknessLighter')};
		:focus {
			position: absolute;
			right: 0;
			max-width: 45rem;
			width: 100%;
			z-index: 200;
		}
		@media screen and (max-width: ${({ theme }) => theme.flexboxgrid.breakpoints.sm}rem) {
			display: none;
		}
	}
`

const TopAlignBar = styled(HorizontalLinkBar)`
	flex: 1 0 auto;
	align-items: flex-start;
	padding: 0;
`
const GrowingListItem = styled(LinkBarItem)`
	flex: 1 0 auto;
`

const StyledLinkBarSearchField = styled(LargeLinkBarSearchField)`
	height: 3.8rem;
	background-color: ${getColor('white')};
	margin-top: 0;
	margin-bottom: 1.5rem;
`

const StyledHorizontalLinkBarWithPadding = styled(HorizontalLinkBar)`
	background-color: ${getColor('darkness')};
	border-bottom: 0.1rem solid ${getColor('darknessLight')};
`

const StyledMobileMenuLink = styled(LinkBarLink)`
	letter-spacing: 0.1rem;
	background-color: ${getColor('darkness')};
	border-bottom: 0.1rem solid ${getColor('darknessLight')};
	display: block;
	padding-left: 4rem;
`

const StyledGetBackMobileMenuLink = styled(LinkBarLink)`
	background-color: ${getColor('darknessLight')};
	text-align: center;
	display: block;
`

const MobileNavigationWrapper = styled(Col)`
	min-width: ${props => props.theme.flexboxgrid.breakpoints.xs}em;
	display: ${props => (props.visible ? 'block' : 'none')};
	padding: 0;
`

const MobileMenuWrapper = styled.div`
	display: none;
	@media screen and (max-width: ${({ theme }) => theme.flexboxgrid.breakpoints.sm}rem) {
		display: block;
	}
`

const DagbladetVideo = dynamic(() =>
	import(
		/* webpackChunkName: "DagbladetVideo" */ '@aller/shiny/src/atoms/SvgIcon/default/DagbladetVideo'
	),
)

const BackNav = dynamic(() =>
	import(/* webpackChunkName: "BackNav" */ '@aller/shiny/src/atoms/SvgIcon/default/BackNav'),
)

const StyledSvgIcon = styled(SvgIcon)`
	margin-left: calc(${getVariable('horizontalBase')} / 4);
`

const SvgIconWrapper = styled.div`
	display: inline-block;
	text-align: center;
	width: ${props => props.width || 'auto'};
	height: ${props => props.height || 'auto'};

	@media (max-width: ${props => props.theme.flexboxgrid.breakpoints.md}em) {
		width: ${props => props['width-sm'] || props.width || 'auto'}rem;
		height: ${props => props['height-sm'] || props.height || 'auto'}rem;
 	}
`

const MobileIconButton = styled(IconButton)`
	padding-left: 0.7rem;
`

const DagbladetVideoHeader = () => {
	const [isVisibleMenu, setVisibleMenu] = useState(false)

	return (
		<Header>
			<TopAlignBar {...TopAlignHeaderProps}>
				<GrowingListItem isListItem>
					<TopAlignBar
						backgroundColor="transparent"
						isTopLevelComponent={false}
						overflow="visible"
						shouldFlexChildren
					>
						<CategoryLink {...LinkBarProps} href="/video">
							<SvgIconWrapper width="20rem" width-sm="12rem" height="65.5px">
								<DagbladetVideo />
							</SvgIconWrapper>
						</CategoryLink>
						<StyledLinkBarLink
							{...LinkBarProps}
							xs={false}
							lg
							linkText="VIRALE"
							href="/video/category/virale"
						/>
						<StyledLinkBarLink
							{...LinkBarProps}
							xs={false}
							sm
							linkText="KJENDIS"
							href="/video/category/kjendis"
						/>
						<StyledLinkBarLink
							{...LinkBarProps}
							xs={false}
							md
							linkText="SPORT"
							href="/video/category/sport"
						/>
						<StyledLinkBarLink
							{...LinkBarProps}
							md
							xs={false}
							linkText="NYHETER"
							href="/video/category/nyheter"
						/>
						{/* <StyledLinkBarLink
							{...LinkBarProps}
							xs={false}
							sm
							linkText="PLUSS"
							href="/video/category/pluss"
						/> */}
						<StyledLinkBarLink
							{...LinkBarProps}
							xs={false}
							sm
							size="small"
							href="https://www.dagbladet.no"
						>
							<SvgIconWrapper width="1.5rem" height="1.5rem">
								<BackNav />
							</SvgIconWrapper>
							<GetBack> Til Dagbladet.no</GetBack>
						</StyledLinkBarLink>
					</TopAlignBar>
				</GrowingListItem>
				<LinkBarLinkDynamicBold
					{...linkProps}
					id="dagbladet-login-button"
					linkText="Logg Inn"
					url="/app/minside"
				/>
				<UserDropDownWrapper selector="dagbladet" />
				<SearchFieldDynamicWidth
					isListItem
					xs={false}
					md
					action="https://www.dagbladet.no/videoarkiv"
					placeholder="Søk"
					backgroundColor="grayTintLighter"
					iconColor="typeLight"
				/>
				<MobileMenuWrapper>
					<MobileIconButton onClick={() => setVisibleMenu(!isVisibleMenu)}>
						MENY <StyledSvgIcon opened={isVisibleMenu} name="hamburger" size={2.5} color="white" />
					</MobileIconButton>
				</MobileMenuWrapper>
			</TopAlignBar>
			<MobileNavigationWrapper visible={isVisibleMenu}>
				<StyledHorizontalLinkBarWithPadding shouldFlexChildren isTopLevelComponent={false}>
					<StyledLinkBarSearchField
						textColor={getColor('darkness')}
						backgroundColor="white"
						isListItem
						fullWidth
						placeholder="Søk"
						action="https://www.dagbladet.no/videoarkiv"
					/>
				</StyledHorizontalLinkBarWithPadding>
				<StyledMobileMenuLink
					{...mobileMenuLinkProps}
					linkText="VIRALE"
					href="/video/category/virale"
				/>
				<StyledMobileMenuLink
					{...mobileMenuLinkProps}
					linkText="KJENDIS"
					href="/video/category/kjendis"
				/>
				<StyledMobileMenuLink
					{...mobileMenuLinkProps}
					linkText="SPORT"
					href="/video/category/sport"
				/>
				<StyledMobileMenuLink
					{...mobileMenuLinkProps}
					linkText="NYHETER"
					href="/video/category/nyheter"
				/>
				{/* <StyledMobileMenuLink
							{...mobileMenuLinkProps}
							linkText="PLUSS"
							href="/video/category/pluss"
						/> */}
				<StyledGetBackMobileMenuLink {...mobileGetBackMenuLinkProps}>
					<SvgIconWrapper height="1.5rem" width="1.5rem">
						<BackNav />
					</SvgIconWrapper>
					<GetBack>Til Dagbladet.no</GetBack>
				</StyledGetBackMobileMenuLink>
			</MobileNavigationWrapper>
		</Header>
	)
}

export default DagbladetVideoHeader
