import React from 'react'
import stripTags from 'striptags'

import getHostname from '../../../utils/get-hostname'
import { percentageToGridValues } from '../../../utils/percentage-to-grid-values'
import fillEmptyGridSizes from './get-grid-sizes'
import { useIsMobile } from '@aller/shared/hooks/useIsMobile'
import { ImageSource } from '../../../types/images'

import { SkistarArticle } from './styles/ArticleCol'
import PlugLink from './styles/PlugLink'
import LabradorLabel from './styles/LabradorLabel'
import ArticlePreviewFooter from './styles/ArticlePreviewFooter'
import Description from './styles/Description'
import Headline from './styles/Headline'
import ArticleKicker from './styles/ArticleKicker'
import Source from './styles/Source'
import ArticlePreviewText from './styles/ArticlePreviewText'
import LazyImage from '../../LazyImage'

import { generateLabradorImageUrls } from '../../../utils/generate-labrador-image-urls'

function getLabelColor(name: string) {
	return name.replace(/\s/g, '').toLowerCase()
}

function calculateRatioByTitle(title: string) {
	const length = stripTags(title, ['strong', 'em']).length

	switch (true) {
		case length <= 54:
			return 1.15
		case length > 54 && length <= 68:
			return 1.36
		case length > 68 && length <= 100:
			return 1.45
		case length >= 100:
			return 1.6
		default:
			return 1.15
	}
}

interface Props {
	className?: string
	loading?: 'eager' | 'auto' | 'lazy'
	attributes?: any
	preventBlur?: boolean
	column: any
	url: string
	kicker: string
	fontSizeOverride?: number
	commercialPartner: string
	logoURL: string
	subtitle?: string
	title: string
	baseImageUrl?: string
	ratio?: number
	dataLabel?: string
	parentWidth?: Array<any>
	hideImage?: boolean
	identifier?: string
	isBlack?: boolean
	image?: any
	[x: string]: any
}

const ContentPlug: React.FC<Props> = ({
	baseImageUrl,
	ratio = 0.5,
	subtitle,
	title,
	dataLabel,
	logoURL,
	commercialPartner,
	column,
	kicker,
	fontSizeOverride,
	url,
	parentWidth = [],
	identifier,
	isBlack = false,
	hideImage = false,
}) => {
	let actualColumnSize = column
	const topnormal = identifier === 'topnormal'
	if (topnormal) {
		if (parentWidth[1] && parentWidth[1].type === 'Labrador_Column') {
			actualColumnSize = percentageToGridValues(parentWidth[1].value)
		}
		column = {
			xs: 12,
			sm: 12,
			md: 12,
			lg: 12,
		}
	}

	let imageSources: ImageSource[] = []
	if (baseImageUrl) {
		imageSources = generateLabradorImageUrls(
			baseImageUrl,
			ratio,
			fillEmptyGridSizes(actualColumnSize),
			true,
		)
	}

	return (
		<SkistarArticle
			isNativeAdvertisement
			className="preview-blink lp_article"
			{...column}
			isBlack={isBlack}
		>
			<PlugLink isNativeAdvertisement isBlack={isBlack} target="_blank" href={url}>
				<ArticleKicker backgroundHoverShade="#fff" isNativeAdvertisement order={1}>
					{kicker}
					{logoURL && <img src={logoURL} alt={commercialPartner} />}
				</ArticleKicker>
				{baseImageUrl && imageSources && imageSources[0] && imageSources[0].url && !hideImage && (
					<div style={{ order: 2 }}>
						<LazyImage
							sources={imageSources}
							alt={stripTags(title)}
							ratio={ratio}
							offset={0}
							src={baseImageUrl}
							preventBlur
							fadeIn={false}
						>
							{imageSources.length === 0 && <Source srcSet={baseImageUrl} />}
							{imageSources.map((source, i) => {
								if (i === 2) {
									return null
								}
								return <Source srcSet={source.url} media={source.media} key={`source-${i}`} />
							})}
						</LazyImage>
					</div>
				)}
				<ArticlePreviewText {...fillEmptyGridSizes(column)}>
					{!!dataLabel && (
						<LabradorLabel labradorLabel={getLabelColor(dataLabel)}>{dataLabel}</LabradorLabel>
					)}
					<Headline
						order={3}
						maxsize={topnormal}
						fontSizeOverride={fontSizeOverride}
						{...fillEmptyGridSizes(actualColumnSize)}
						dangerouslySetInnerHTML={{ __html: title }} // eslint-disable-line react/no-danger
					/>
					{subtitle && (
						<Description
							isBlack={isBlack}
							{...fillEmptyGridSizes(column)}
							order={4}
							itemProp="description"
						>
							{subtitle}
						</Description>
					)}
				</ArticlePreviewText>
			</PlugLink>
		</SkistarArticle>
	)
}

export default ContentPlug
