import React from 'react'
import propTypes from 'prop-types'
import { connect } from 'react-redux'
import { bindActionCreators } from 'redux'
import { LinkBarLink } from '@aller/shiny/src/atoms/LinkBarLink'
import { HorizontalLinkBar } from '@aller/shiny/src/molecules/HorizontalLinkBar'

import { getColor } from '@aller/shiny/src/utils/get-color'

import { withTheme } from 'styled-components'
import logClick from '../services/log-click'
import { getArticles, showSettings, setActiveCategory } from '../actions'

import SettingsButton from './SettingsButton'

function mapDispatchToProps(dispatch) {
	return {
		actions: bindActionCreators(
			{
				getArticles,
				showSettings,
				setActiveCategory
			},
			dispatch
		)
	}
}

const CategoryButton = connect(
	mapStateToProps,
	mapDispatchToProps
)(LinkBarLink)

class CategoryNavigationBar extends React.Component {
	handleNavButtonClick(event, { name }) {
		event.preventDefault()
		event.stopPropagation()

		const {
			actions: { showSettings, getArticles, setActiveCategory },
			selectedCounties
		} = this.props

		logClick(`newsstream_menu_${name}`)
		if (name === 'lokale' && selectedCounties.length === 0) {
			setActiveCategory(name)
			showSettings()
		} else {
			getArticles(name, 'navigation')
		}
	}

	renderCategoryButtons() {
		const { categorySet, currentCategory } = this.props
		return categorySet
			.map((category, index) => {
				return (
					<CategoryButton
						isActive={currentCategory === category.name}
						key={index}
						linkText={category.title}
						activeBackgroundColor="grayTintLighter"
						url={`#${category.name}`}
						useUnderline={false}
						onClick={event =>
							this.handleNavButtonClick(event, {
								name: category.name
							})
						}
					/>
				)
			})
			.concat(<SettingsButton key="settingsbutton_newsstreamnav" />)
	}

	render() {
		return (
			<HorizontalLinkBar
				shouldHavePadding={false}
				isTopLevelComponent={false}
				background={getColor('white')(this.props)}
			>
				{this.renderCategoryButtons()}
			</HorizontalLinkBar>
		)
	}
}

CategoryNavigationBar.propTypes = {
	actions: propTypes.shape({
		getArticles: propTypes.func.isRequired,
		showSettings: propTypes.func.isRequired,
		setActiveCategory: propTypes.func.isRequired
	}).isRequired,
	categorySet: propTypes.arrayOf(propTypes.object),
	currentCategory: propTypes.string.isRequired,
	selectedCounties: propTypes.arrayOf(propTypes.string).isRequired
}

CategoryNavigationBar.defaultProps = {
	categorySet: []
}

function mapStateToProps(state) {
	return {
		categorySet: state.categorySet,
		currentCategory: state.currentCategory,
		selectedCounties: state.settings.localNews.selectedCounties
	}
}

export default withTheme(
	connect(
		mapStateToProps,
		mapDispatchToProps
	)(CategoryNavigationBar)
)
