import React from 'react'
import propTypes from 'prop-types'
import { connect } from 'react-redux'
import { bindActionCreators } from 'redux'
import classnames from 'classnames'

import getOpenerRel from '../../../utils/get-opener-rel'
import makeLinkpulseClassName from '../../../utils/make-linkpulse-class-name'

import { Article, Link, Image, Title, Description } from '../styles/article-card'

const ArticleCard = (props, context) => {
	const { articleCardData, currentCategory } = props

	if (!articleCardData) {
		return null
	}

	const {
		title = '',
		url = '',
		fields: { image = '', description = '' }
	} = articleCardData

	const target = '_blank'
	const openerRel = getOpenerRel(target)

	const rel = classnames('bookmark', openerRel)

	const urlEncodedImage = encodeURIComponent(image)
	const length = 150 // set to the number of characters you want to keep
	let truncatedDescription = description
	if (description.length >= length) {
		truncatedDescription = description.substring(0, Math.min(length, description.length))
		truncatedDescription = `${truncatedDescription} ...`
	}
	return (
		<Article>
			<Link
				href={url}
				target={target}
				rel={rel}
				className={makeLinkpulseClassName(`newsstream_card_${currentCategory}`)}
			>
				{image && (
					<Image
						src={`https://sol.no/app/cimg/t_sol-mobile-full,w_200/${urlEncodedImage}`}
						title={title}
						alt=""
					/>
				)}
				<Title>{title}</Title>
				<Description>{truncatedDescription}</Description>
			</Link>
		</Article>
	)
}

ArticleCard.propTypes = {
	articleCardData: propTypes.shape({
		title: propTypes.string,
		url: propTypes.string,
		fields: propTypes.shape({
			image: propTypes.string,
			description: propTypes.string
		})
	}),
	currentCategory: propTypes.string.isRequired
}

ArticleCard.defaultProps = {
	articleCardData: {
		title: '',
		url: '',
		fields: {
			image: '',
			description: ''
		}
	}
}

function mapStateToProps(state) {
	return {
		articleCardData: state.articleCardData,
		currentCategory: state.currentCategory
	}
}

function mapDispatchToProps(dispatch) {
	return {
		actions: bindActionCreators({}, dispatch)
	}
}

export default connect(
	mapStateToProps,
	mapDispatchToProps
)(ArticleCard)
