import fetchArticles from '../services/fetch-articles'
import hasStorage from '../../../utils/has-storage'

export const POPULATE_DATA_PENDING = 'POPULATE_DATA_PENDING'
export const POPULATE_DATA_SUCCESS = 'POPULATE_DATA_SUCCESS'
export const POPULATE_ERROR = 'POPULATE_ERROR'
export const CLEAR_ARTICLE_DATA = 'CLEAR_ARTICLE_DATA'
export const POPULATE_CARD_DATA_SUCCESS = 'POPULATE_CARD_DATA_SUCCESS'

export const SET_ACTIVE_CATEGORY = 'SET_ACTIVE_CATEGORY'
export const UPDATE_CATEGORY_SET = 'UPDATE_CATEGORY_SET'

export const UPDATE_SETTINGS = 'UPDATE_SETTINGS'
export const DISPLAY_SETTINGS = 'DISPLAY_SETTINGS'

export const PAGINATION_COUNTER = 'PAGINATION_COUNTER'
export const PAGINATION_OFFSET = 'PAGINATION_OFFSET'

function populateCardDataSuccess(articleData) {
	return {
		type: POPULATE_CARD_DATA_SUCCESS,
		articleData,
	}
}

function populateDataPending() {
	return {
		type: POPULATE_DATA_PENDING,
	}
}

function populateDataSuccess(articles) {
	return dispatch => {
		dispatch({
			type: POPULATE_DATA_SUCCESS,
			articles,
		})

		const offset = articles.length
		dispatch(paginationOffset(offset))
	}
}

function populateError(error) {
	return {
		type: POPULATE_ERROR,
		error,
	}
}

export function clearArticleData() {
	return {
		type: CLEAR_ARTICLE_DATA,
	}
}

export function setActiveCategory(newCategory) {
	return (dispatch, getState) => {
		const reduxState = getState()
		const oldCategory = reduxState.currentCategory

		dispatch(updateOnlyOneHostKeyInACateogry('')) // The old category shall reset 'onlyOneHost' first

		if (newCategory !== oldCategory) {
			dispatch({
				type: SET_ACTIVE_CATEGORY,
				previousCategory: oldCategory,
				currentCategory: newCategory,
			})
		}
	}
}

function updateOnlyOneHostKeyInACateogry(hostname) {
	// hostname will be '' when 'all newspapers' is chosen
	return (dispatch, getState) => {
		const { currentCategory, categorySet } = getState()

		const newCategorySet = categorySet.map(category => {
			if (category.name === currentCategory) {
				category.onlyOneHost = hostname
			}
			return category
		})
		dispatch(updateCategorySet(newCategorySet))
	}
}

export function setActiveNewsPaperInCategory(hostName) {
	return (dispatch, getState) => {
		const { currentCategory } = getState()

		dispatch(updateOnlyOneHostKeyInACateogry(hostName))
		dispatch(getArticles(currentCategory, 'onlyOneHost'))
	}
}

export function showSettings() {
	return dispatch => {
		if (!hasStorage()) {
			dispatch(populateError(getErrorMessage('noLocalStorage')))
			return
		}

		dispatch({
			type: DISPLAY_SETTINGS,
			displaySettings: true,
		})
	}
}

export function hideSettings() {
	return {
		type: DISPLAY_SETTINGS,
		displaySettings: false,
	}
}

export function updateCategorySet(categorySet) {
	return {
		type: UPDATE_CATEGORY_SET,
		categorySet,
	}
}

export function updateSettings(selectedCounties) {
	return (dispatch, getState) => {
		const harvesterFormatted = selectedCounties.toString()

		const reduxState = getState()
		const { categorySet } = reduxState
		const newCategorySet = categorySet.map(category => {
			if (category.name === 'lokale') {
				category.includeAnyTags = harvesterFormatted
			}
			return category
		})
		dispatch(updateCategorySet(newCategorySet))

		const { settings } = reduxState
		settings.localNews.selectedCounties = selectedCounties
		dispatch({
			type: UPDATE_SETTINGS,
			settings,
		})
	}
}

export function storeSettings(selectedCounties) {
	return dispatch => {
		if (hasStorage()) {
			const selectedCountiesStringified = JSON.stringify(selectedCounties)
			localStorage.setItem('sol-local-news-sources', selectedCountiesStringified)
		} else {
			dispatch(populateError(getErrorMessage('noLocalStorage')))
		}

		dispatch(updateSettings(selectedCounties))
	}
}

export function fetchSettings() {
	return dispatch => {
		dispatch(populateDataPending())
		if (hasStorage()) {
			const selectedCountiesStringified = localStorage.getItem('sol-local-news-sources')
			if (selectedCountiesStringified) {
				const data = JSON.parse(selectedCountiesStringified)
				dispatch(updateSettings(data))
			}
		}
	}
}

export function setArticleCardData(index) {
	return (dispatch, getState) => {
		const reduxState = getState()
		try {
			const articleData = reduxState.articles[index]
			dispatch(populateCardDataSuccess(articleData))
		} catch (e) {
			dispatch(populateError(e))
		}
	}
}

export function getArticles(category, type = '') {
	return (dispatch, getState) => {
		dispatch(populateDataPending())

		if (type !== 'onlyOneHost') {
			dispatch(setActiveCategory(category))
		}

		if (type !== 'pagination') {
			dispatch(paginationCounter('reset'))
			dispatch(paginationOffset(0))
			dispatch(clearArticleData())
		}

		const {
			fetchArticles: { limit, timeout, offset },
			categorySet,
		} = getState()

		const options = {
			categoryData: getCategoryDataByName(category, categorySet),
			limit,
			timeout,
			offset,
		}

		fetchArticles(options)
			.then(newArticles => {
				const { articles: existingArticles } = getState()

				existingArticles.length === 0
					? dispatch(populateDataSuccess(newArticles))
					: dispatch(populateDataSuccess(existingArticles.concat(newArticles)))
			})
			.catch(err => {
				dispatch(populateError(getErrorMessage(err)))
			})
	}
}

function getCategoryDataByName(categoryName, categorySet) {
	const category = categorySet.filter(cat => cat.name === categoryName)
	if (category.length <= 0) {
		throw new Error(`NewsStream: No data to build search-string for ${categoryName}`)
	}
	return category[0]
}

function getErrorMessage(error) {
	if (error.toString().includes('timeout')) {
		return (
			'Beklager, det tok for lang tid å hente dataene.\n' +
			'Vennligst forsøk på nytt ved å klikke på en av kategoriene.'
		)
	}

	if (error.toString().includes('noLocalStorage')) {
		return (
			'Din nettleser støtter ikke lagring (vi bruker localStorage).\n' +
			'Vi kan derfor desverre ikke ta vare på valgene du gjør.\n' +
			'Du kan antagelig slå det på igjen i nettleserens innstillinger.'
		)
	}

	return (
		'Beklager, det har skjedd en feil.\n' +
		'Du kan forsøke igjen ved å klikke på en av kategoriene.'
	)
}

function paginationOffset(offset) {
	return {
		type: PAGINATION_OFFSET,
		offset,
	}
}

export function paginationCounter(task, isDesktop) {
	return (dispatch, getState) => {
		const reduxState = getState()
		const previousNumberOfPages = reduxState.pagination.pages
		const { currentCategory } = reduxState

		let pages
		switch (task) {
			case 'inc':
				pages = previousNumberOfPages + 1
				if (shouldFetchMoreArticles(reduxState, isDesktop)) {
					dispatch(getArticles(currentCategory, 'pagination'))
				}
				break
			case 'dec':
				if (previousNumberOfPages > 1) {
					pages = previousNumberOfPages - 1
				} else {
					pages = 1
				}
				break
			default:
				pages = previousNumberOfPages
		}

		dispatch({
			type: PAGINATION_COUNTER,
			pages,
		})
	}
}

function shouldFetchMoreArticles(reduxState, isDesktop) {
	const {
		articles,
		pagination: {
			articlesPerPage: { mobile, desktop },
			pages,
		},
	} = reduxState

	const articlesPerPage = isDesktop ? desktop : mobile
	const numArticles = articles.length

	return numArticles < articlesPerPage * (pages + 1) // N.B: +1 because 'pages' is not updated yet
}
