import React from 'react'
import Head from 'next/head'

type Props = {
	title: string
	description: string
	noIndex: boolean
	noFollow: boolean
	domain: string
	isTablet?: boolean
	publishedURL: string
	section?: string
	name: string
	twitter?: string
	facebookPages?: string[]
	siteName: string
	children?: Array<React.ReactElement> | React.ReactElement
}

const Metadata = ({
	title = '',
	description = '',
	noIndex = false,
	noFollow = false,
	domain = '',
	publishedURL = '',
	section = '',
	name = '',
	twitter = undefined,
	facebookPages = undefined,
	siteName,
	isTablet = false,
	children
}: Props) => {
	const robots = []
	if (noFollow) {
		robots.push('nofollow')
	}
	if (noIndex) {
		robots.push('noindex')
	}

	const year = new Date().getFullYear()

	return (
		<Head>
			{robots.length > 0 && <meta name="robots" content={robots.join(',')} />}
			<link rel="manifest" href="/manifest.json" />
			<meta
				name="copyright"
				content={`Copyright (c) ${year} Aller Media AS, all rights reserved.`}
			/>
			<meta name="unispring_s" content="allerinternett" />
			<meta name="unispring_cp" content={`aller/${siteName}/${section}`} />
			<meta name="title" content={title} />
			<meta name="description" content={description} />
			<meta property="og:title" content={title} />
			<meta property="og:url" content={`https://${domain}${publishedURL}`} />
			<meta property="og:type" content="webpage" />
			<meta property="og:site_name" content={name} />
			{isTablet && <meta name="viewport" content="width=1024,user-scalable=yes, shrink-to-fit=yes" />}
			<meta
				property="og:image"
				content={`https://styleguide.${siteName}.no/assets/favicon/${siteName}/favicon-228.png`}
			/>
			<meta property="og:description" content={description} />
			{facebookPages && facebookPages.map(id => (
				<meta property="fb:pages" content={id} key={`fb-meta-pages-${id}`} />
			))}
			{twitter && (
				<>
					<meta name="twitter:card" content="summary" />
					<meta name="twitter:site" content={`@${twitter}`} />
					<meta name="twitter:title" content={title} />
					<meta name="twitter:description" content={description} />
				</>
			)}

			<meta property="lp:url" content={`https://${domain}${publishedURL}`} />
			<meta property="page:site_section" content={section} />
			<title>{title}</title>
			{children}
			<link rel="dns-prefetch" href="https://adx.adform.net" />
			<link rel="dns-prefetch" href="https://securepubads.g.doubleclick.net" />
			<link rel="dns-prefetch" href="https://fonts.gstatic.com" />
			<link rel="dns-prefetch" href="https://www.googletagmanager.com" />
			<link rel="dns-prefetch" href="https://www.googletagservices.com" />
			<link rel="dns-prefetch" href="https://bam.nr-data.net" />
			<link rel="dns-prefetch" href="https://csi.gstatic.com" />
			<link rel="dns-prefetch" href="https://ads.avocet.io" />
			<link rel="dns-prefetch" href="https://pixel-sync.sitescout.com" />
			<link rel="dns-prefetch" href="https://dclk-match.dotomi.com" />
			<link rel="dns-prefetch" href="https://bid.g.doubleclick.net" />
			<link rel="dns-prefetch" href="https://pagead2.googlesyndication.com" />
			<link rel="dns-prefetch" href="https://tpc.googlesyndication.com" />
			<link rel="preconnect" href="https://currency.prebid.org" crossOrigin="anonymous" />
			<link rel="preconnect" href="https://adservice.google.com" crossOrigin="anonymous" />
			<link rel="preconnect" href="https://ads.rubiconproject.com" crossOrigin="anonymous" />
			<link rel="preconnect" href="https://s.update.rubiconproject.com" crossOrigin="anonymous" />
			<link rel="preconnect" href="https://fastlane.rubiconproject.com" crossOrigin="anonymous" />
			<link rel="preconnect" href="https://code3.adtlgc.com" crossOrigin="anonymous" />
			<link rel="preconnect" href="https://ib.adnxs.com" crossOrigin="anonymous" />
			<link rel="preconnect" href="https://platform.instagram.com" crossOrigin="anonymous" />

		</Head>
	)
}

export default Metadata
