import React, { useMemo } from 'react'
import styled from 'styled-components'
import { css } from 'styled-components'
import { Col } from '@aller/shiny/src/atoms/Col'
import { PlugHeading } from '@aller/shiny/src/atoms/PlugHeading'
import { getColor } from '@aller/shiny/src/utils/get-color'
import { getVariable } from '@aller/shiny/src/utils/get-variable'

import { Column } from '../FlexCol/types'
import { generateLabradorImageUrls } from '../../utils/generate-labrador-image-urls'
import { generatePastelHSLFromString } from '../../utils/color-tools'
import LazyImage from '../LazyImage'

type TitleType = 'small' | 'medium' | 'large' | 'giga'

interface TitleProps {
	type?: TitleType
	xs?: Number
}

const Title = styled(PlugHeading)<TitleProps>`
	font-variant-ligatures: no-common-ligatures;
	font-style: normal;
	font-weight: ${getVariable('headingsWeight')};
	text-rendering: optimizeLegibility;
	font-family: ${getVariable('headingsFont')};
	-webkit-font-smoothing: antialiased;
	-moz-osx-font-smoothing: grayscale;

	&:hover {
		text-decoration: underline;
	}

	font-size: 3.4rem;
	line-height: 1.3;
	${(props: TitleProps) =>
		props.xs &&
		props.xs < 12 &&
		css`
			font-size: 2.2rem;
			line-height: 1.3;
		`}

	@media screen and (min-width: ${props => (props.theme as any).flexboxgrid.breakpoints.sm}rem) {
		${(props: TitleProps) => {
			let fontSize = '3.4rem'
			let lineHeight = '4rem'

			if (props.type === 'small') {
				fontSize = '2.4rem'
				lineHeight = '2.8rem'
			}

			if (props.type === 'medium') {
				fontSize = '2.8rem'
				lineHeight = '3.2rem'
			}

			if (props.type === 'large') {
				fontSize = '3.4rem'
				lineHeight = '4rem'
			}

			if (props.type === 'giga') {
				fontSize = '4rem'
				lineHeight = '4.6rem'
			}

			return css`
				font-size: ${fontSize};
				line-height: ${lineHeight};
			`
		}};
	}
`

const StyledSubtitle = styled(PlugHeading)<TitleProps>`
	font-variant-ligatures: no-common-ligatures;
	font-style: normal;
	font-weight: 300;
	text-rendering: optimizeLegibility;
	font-family: ${getVariable('mainFont')};
	-webkit-font-smoothing: antialiased;
	-moz-osx-font-smoothing: grayscale;
	font-size: 1.7rem;
	line-height: 2rem;
	margin-bottom: 0.8rem;
`

const Subtitle = StyledSubtitle.withComponent('h2')

const Anchor = styled.a`
	text-decoration: none;
	color: ${props => getColor('primary')(props)};
`

const CategoryHeading = styled.h2`
	margin-top: 1.2rem;
	margin-bottom: 1rem;
	letter-spacing: 0.1em;
	text-transform: uppercase;
`
const CategorySpan = styled.span`
	position: relative;
	padding-bottom: 5px;
	display: inline-block;

	font-size: 1.3rem;
	@media screen and (min-width: ${props => (props.theme as any).flexboxgrid.breakpoints.sm}rem) {
		font-size: 1.5rem;
	}

	&:after {
		content: '';
		position: absolute;
		display: inline;
		left: 0;
		bottom: 0;
		width: 100%;
		height: 0.2rem;
		border-bottom: 0.2rem solid ${props => props.color};
	}
`

const ContentMarking = styled.span`
	position: relative;
	padding-bottom: 5px;
	display: inline-block;
	margin-bottom: 1.6rem;
	letter-spacing: 0.1em;
	text-transform: uppercase;

	font-size: 1.1rem;
	&::before {
		content: 'Annonse';
		margin-right: 0.7rem;
		color: ${getColor('white')};
		padding: calc(${getVariable('verticalBase')} / 4);
		text-transform: uppercase;
		background: black;
		display: inline-block;
	}
`

const ContentMarketingBold = styled.span`
	font-weight: 700;
`

const Category = ({ color, children }: { color: string; children: any }) => (
	<CategoryHeading>
		<CategorySpan color={color}>{children}</CategorySpan>
	</CategoryHeading>
)

const ArticleCol = Col.withComponent('article')
const Article = styled(ArticleCol)`
	margin-bottom: 1.5rem;
	${props =>
		props.commercialPartner &&
		css`
			&::before,
			&::after {
				content: '';
				display: block;
				height: 0.3rem;
				background-color: ${getColor('darkness')(props)};
				padding-left: calc(${getVariable('horizontalBase')(props)} / 2);
				padding-right: calc(${getVariable('horizontalBase')(props)} / 2);
				margin-bottom: calc(${getVariable('horizontalBase')(props)} / 2);
			}
		`}
`

interface Props {
	className?: string
	loading?: 'eager' | 'auto' | 'lazy'
	attributes?: any
	preventBlur?: boolean
	baseImageUrl?: string
	ratio?: number
	image?: any
	url: string
	title: string
	column?: Column
	category?: string
	fromFront?: boolean
	subtitle?: string
	preset?: string
	commercialPartner?: string
	isVideo?: boolean
	type?: TitleType
	[x: string]: any
}

const LoganPlug: React.FC<Props> = ({
	className = '',
	attributes = { className: '' },
	preventBlur = true,
	baseImageUrl,
	ratio = 0.5,
	image,
	url,
	loading = 'lazy',
	type,
	title,
	subtitle,
	category = '',
	preset = '',
	fromFront = false,
	commercialPartner = null,
	column = {
		xs: 12,
	},
	...rest
}) => {
	let sources: any = []

	const isVideo = category === 'video' || category === 'se tv'

	/* eslint-disable react-hooks/rules-of-hooks */
	if (baseImageUrl) {
		sources = useMemo(() => generateLabradorImageUrls(baseImageUrl, ratio, column, true), [
			baseImageUrl,
			column,
			ratio,
		])
	}

	let imageUrl = sources[0].url
	if (sources.length > 2) [(imageUrl = sources[2].url || image)]
	const categoryHSLA = useMemo(() => generatePastelHSLFromString(category), [category])

	return (
		<Article
			data-testid="logan-plug"
			{...column}
			{...rest}
			className={`${className} preview-blink`}
			data-cxense_tag="cerebro"
			data-cxence_widget={preset}
			commercialPartner={commercialPartner}
		>
			<Anchor href={url}>
				<LazyImage
					sources={sources}
					src={imageUrl}
					alt={title}
					ratio={ratio}
					loading={loading}
					isVideo={isVideo}
				/>
				{category && <Category color={categoryHSLA}>{category}</Category>}
				<Title type={type} xs={column.xs}>
					{title}
				</Title>
				{fromFront && subtitle && <Subtitle>{subtitle}</Subtitle>}
				{commercialPartner && (
					<ContentMarking>
						i samarbeid med <ContentMarketingBold>{commercialPartner}</ContentMarketingBold>
					</ContentMarking>
				)}
			</Anchor>
		</Article>
	)
}
;(LoganPlug as any).whyDidYouRender = true

export default LoganPlug
