/* eslint-disable global-require */
import React, { useEffect, createContext, useReducer, useContext } from 'react'
import PropTypes from 'prop-types'
import ulog from 'ulog'

const log = ulog('ads')

const defaultAdsState = {
	showSkyscraper: false,
	showModal: false,
	arcAds: null,
}

const SlotsStoreContext = createContext({})
const AdsStoreContext = createContext(defaultAdsState)
const AdsDispatchContext = createContext(null)

export const gptSlots = {}

const slotReducer = (state, action) => {
	switch (action.type) {
		case 'ACTIVATE_SHOW_COG':
			if (state[action.id] && state[action.id].showCog === true) return state
			return {
				...state,
				[action.id]: {
					...state[action.id],
					showCog: true,
				},
			}
		default:
			return state
	}
}

const adsReducer = (state, action) => {
	switch (action.type) {
		case 'SHOW_MODAL':
			if (state.showModal === true) return state
			return {
				...state,
				showModal: true,
			}
		case 'HIDE_MODAL':
			if (state.showModal === false) return state
			return {
				...state,
				showModal: false,
			}
		case 'SHOW_SKYSCRAPER':
			if (state.showSkyscraper === true) return state
			return {
				...state,
				showSkyscraper: true,
			}
		case 'ARCADS_READY':
			log.info('ArcAds ready!')
			if (state.arcAds === action.arcAds) return state
			return {
				...state,
				arcAds: action.arcAds,
			}
		case 'REGISTER_AD':
			if (state.arcAds) {
				log.info(`Registering ad ${action.options.id}`)
				state.arcAds.registerAd(action.options)
			}
			return state
		default:
			return state
	}
}

const AdProvider = ({ children }) => {
	const [slots, slotsDispatch] = useReducer(slotReducer, {})
	const [state, dispatch] = useReducer(adsReducer, defaultAdsState)

	const handleEvent = event => {
		const id = event.slot.getSlotElementId()
		gptSlots[id] = event.slot

		if (!event.isEmpty) {
			slotsDispatch({
				type: 'ACTIVATE_SHOW_COG',
				id,
			})

			if (id === 'ad-topbanner1') {
				if (event.size && event.size[0] < 1000) {
					dispatch({
						type: 'SHOW_SKYSCRAPER',
					})
				}
			}
		}
	}

	useEffect(() => {
		async function loadAdLibraries() {
			const prebid = await import('prebid.js')
			import('prebid.js/modules/currency')
			import('prebid.js/modules/rubiconBidAdapter')
			import('prebid.js/modules/appnexusBidAdapter')
			import('prebid.js/modules/adformBidAdapter')
			import('prebid.js/modules/adformOpenRTBBidAdapter')
			import('prebid.js/modules/dfpAdServerVideo')
			log.info('Loaded prebid.js modules. Processing queue...')
			prebid.default.processQueue()
			const ArcAds = await import('arcads')
			log.info('Loaded arcads')
			const arcAds = new ArcAds.ArcAds(
				{
					dfp: {
						id: '8578',
					},
					bidding: {
						prebid: {
							enabled: true,
							config: {
								bidderSequence: 'random',
								enableSendAllBids: false,
								currency: {
									adServerCurrency: 'NOK',
								},
								priceGranularity: {
									buckets: [
										{
											min: 0,
											max: 400,
											increment: 0.1,
										},
									],
								},
							},
							timeout: 1000,
						},
					},
				},
				handleEvent,
			)
			dispatch({ type: 'ARCADS_READY', arcAds })
		}
		loadAdLibraries()
	}, [])

	return (
		<SlotsStoreContext.Provider value={slots}>
			<AdsStoreContext.Provider value={state}>
				<AdsDispatchContext.Provider value={dispatch}>{children}</AdsDispatchContext.Provider>
			</AdsStoreContext.Provider>
		</SlotsStoreContext.Provider>
	)
}

export const useSlotsStore = () => useContext(SlotsStoreContext)
export const useAdsStore = () => useContext(AdsStoreContext)
export const useAdsDispatch = () => useContext(AdsDispatchContext)

AdProvider.whyDidYouRender = true

AdProvider.propTypes = {
	children: PropTypes.node.isRequired,
}

export default AdProvider
