import Cookies from 'js-cookie'
import React from 'react'

import { SmallRoundedButton } from '@aller/shiny/src/atoms/RoundedButtons'
import {
	StyledWeatherContainer,
	StyledWeatherData,
	StyledWeatherUpcoming,
	StyledWeatherToday,
	StyledAboveZero,
	StyledBelowZero
} from '../styles/WeatherStyles'

import mathRound from '../../../utils/math-round'

const weekdays = ['Søndag', 'Mandag', 'Tirsdag', 'Onsdag', 'Torsdag', 'Fredag', 'Lørdag']

class Weather extends React.Component {
	constructor() {
		super()

		this.state = { weather: {} }
	}

	componentDidMount() {
		this.fetchWeather()
	}

	fetchWeather() {
		let query = 'weather?locationId=norge/oslo/oslo/oslo'

		const weatherCookie = Cookies.get('weather-location')

		if (weatherCookie) {
			const data = JSON.parse(weatherCookie)
			query = `weather?locationId=${data.id}`
		}

		weatherApiRequest(query).then(data => this.setState({ weather: data }))
	}

	renderWeather() {
		const { weather } = this.state

		if (!(weather && weather.forecast && weather.location && weather.forecast.tabular[0])) {
			return null
		}

		const weatherData = weather.forecast.tabular
			.filter(pickCorrectPeriods)
			.slice(0, 4)
			.map((data, i) => formatWeatherData(data, i, weather.location))

		return <StyledWeatherData>{weatherData}</StyledWeatherData>
	}

	render() {
		return (
			<StyledWeatherContainer className="weather lp_fp_weather">
				<h4>Været i dag</h4>
				<a className="settings" href="https://sol.no/vaer/innstillinger">
					Endre sted
				</a>
				{this.renderWeather()}
				<SmallRoundedButton href="https://sol.no/vaer">Se været flere steder</SmallRoundedButton>
			</StyledWeatherContainer>
		)
	}
}

function formatWeatherData(data, i, location) {
	if (!data) {
		return null
	}

	const weekday = new Date().getDay()
	const temperature = mathRound(data.temperature.value)
	const temperatureData =
		temperature > 0 ? (
			<StyledAboveZero className="temperature">
				{temperature}
				°C
			</StyledAboveZero>
		) : (
			<StyledBelowZero className="temperature">
				{temperature}
				°C
			</StyledBelowZero>
		)
	const imageUrl =
		i === 0
			? `https://sol.no/s/img/h_38,w_38/static/weather_symbol/${data.symbol.numberEx}.png`
			: `https://sol.no/s/img/h_30,w_30/static/weather_symbol/${data.symbol.numberEx}.png`
	const image = <img alt={data.symbol.name} src={imageUrl} />

	if (i === 0 && location) {
		return (
			<StyledWeatherToday key={`weather-box-data-${i}`}>
				<div className="location">
					<span className="place">{location.place}</span>
					<span className="municipality">{location.municipality}</span>
				</div>
				<span className="symbol">{image}</span>
				{temperatureData}
			</StyledWeatherToday>
		)
	}

	return (
		<StyledWeatherUpcoming key={`weather-box-data-${i}`}>
			<span className="day">{i === 1 ? 'I morgen' : wrapIndex(weekday + i, weekdays)}</span>
			<span className="symbol">{image}</span>
			{temperatureData}
		</StyledWeatherUpcoming>
	)
}

function pickCorrectPeriods(data, i) {
	const date = new Date().toISOString().slice(0, 10)

	// For today, grab the current period.
	if (data.time.from.slice(0, 10) === date && i === 0) {
		return true
	}

	// For upcoming dates, pick the second period.
	if (data.time.period === 2) {
		return true
	}

	return false
}

export default Weather
