import React, { useEffect, memo } from 'react'
import styled, { css } from 'styled-components'
import LazyLoad from 'vanilla-lazyload'

declare global {
	interface Document {
		lazyLoadInstance: any
	}
}

const lazyloadConfig = {
	elements_selector: '.lazyload',
	// use_native: true - Turns on native chrome lazyload, seems broken, need to investigate
}

if (process.browser) {
	if (!document.lazyLoadInstance) {
		document.lazyLoadInstance = new LazyLoad(lazyloadConfig)
	}
}

const Picture = styled.picture`
	position: absolute;
	top: 0;
	bottom: 0;
	right: 0;
	left: 0;
`

const Image: any = styled.img`
	object-fit: cover;
	width: 100%;
	height: 100%;
	transition: opacity 0.7s;

	&:not([src]) {
		visibility: hidden;
	}

	&[data-src],
	&[data-srcset] {
		display: block;
		min-height: 1px;
	}

	&.lazyload {
		opacity: 0;

		&.loaded {
			opacity: 1;
		}
	}
`
interface FigureProps {
	ratio: number
	isVideo?: boolean
}

const Figure = styled.figure<FigureProps>`
	position: relative;
	background-color: #ececec;
	display: block;
	padding-bottom: ${(props: FigureProps) => `${props.ratio * 100}%`};
	width: 100%;
	overflow: hidden;
	margin: 0;

	${(props: FigureProps) =>
		props.isVideo &&
		css`
			&:before {
				content: '';
				width: 0px;
				height: 0px;
				border-radius: 0px;
				position: absolute;
				z-index: 11;
				bottom: 15px;
				left: 16px;
				border-top: 6px solid transparent;
				border-bottom: 5px solid transparent;
				border-left: 9px solid #222;
			}

			&:after {
				font-weight: 500;
				text-transform: uppercase;
				letter-spacing: 0.1rem;
				line-height: 0;
				font-size: 0.8rem;
				background-color: #fff;
				padding: 13px 7px 12px 25px;
				position: absolute;
				bottom: 8px;
				left: 8px;
				content: '';
				height: 25px;
				width: 25px;
				padding: 0;
				box-shadow: 0 1px 3px rgba(0, 0, 0, 0.2);
			}
		`}
`

interface Props {
	alt: string
	src: string
	ratio: number
	sources: any[]
	loading?: 'auto' | 'lazy' | 'eager'
	isVideo?: boolean
}

const LazyImage: any = ({ alt, src, ratio, sources, loading = 'lazy', ...rest }: Props) => {
	useEffect(() => {
		document.lazyLoadInstance.update()
	}, [])

	return (
		<Figure ratio={ratio} {...rest}>
			<Picture>
				{sources.map(source => {
					if (loading !== 'eager') {
						return (
							<source
								data-srcset={source.url}
								media={source.media}
								key={`${source.url}${source.media}`}
							/>
						)
					}
					return (
						<source srcSet={source.url} media={source.media} key={`${source.url}${source.media}`} />
					)
				})}

				{loading !== 'eager' ? (
					<>
						<noscript>
							{sources.map(source => (
								<source
									srcSet={source.url}
									media={source.media}
									key={`${source.url}${source.media}`}
								/>
							))}
							<Image
								src={src}
								alt={alt}
								itemProp="image"
								className="lazyload"
								loading={loading}
								style={{ opacity: 1 }}
							/>
						</noscript>
						<Image alt={alt} itemProp="image" data-src={src} className="lazyload" />
					</>
				) : (
					<Image alt={alt} itemProp="image" src={src} />
				)}
			</Picture>
		</Figure>
	)
}

export default memo(LazyImage)
