import React from 'react'
import PropTypes from 'prop-types'
import { ResultPagination, Pagination } from './searchPagination'

const propTypes = {
	results: PropTypes.arrayOf(
		PropTypes.shape({
			title: PropTypes.string,
			date: PropTypes.string,
			page: PropTypes.number,
			catalog: PropTypes.number,
			thumb: PropTypes.string,
			snippet: PropTypes.string,
		}),
	),
	total: PropTypes.number.isRequired,
	pageSize: PropTypes.number,
	currentPage: PropTypes.number.isRequired,
}

const defaultProps = {
	pageSize: 20,
	results: [],
}

const updateQueryParam = (name, value) => {
	const url = new URL(window.location.href)
	const queryString = url.search
	const searchParams = new URLSearchParams(queryString)
	searchParams.delete(name)
	searchParams.append(name, value)
	url.search = searchParams.toString()
	return url
}

class EavisSearchPagination extends React.Component {
	constructor(props) {
		super(props)
		this.state = {
			pager: {},
		}
	}

	componentDidMount() {
		const { results, currentPage, total, pageSize } = this.props
		if (results && results.length && currentPage) {
			this.setState({ pager: this.getPager(total, currentPage, pageSize) })
		}
	}

	getPager = (totalItems, currentPage, pageSize) => {
		// default to first page
		currentPage = currentPage || 1

		// default page size is 20
		pageSize = pageSize || 20

		// calculate total pages
		const totalPages = Math.ceil(totalItems / pageSize)

		let startPage
		let endPage
		if (totalPages <= 10) {
			// less than 10 total pages so show all
			startPage = 1
			endPage = totalPages
		} else {
			// more than 10 total pages so calculate start and end pages
			if (currentPage <= 6) {
				startPage = 1
				endPage = 10
			} else if (currentPage + 4 >= totalPages) {
				startPage = totalPages - 9
				endPage = totalPages
			} else {
				startPage = currentPage - 5
				endPage = currentPage + 4
			}
		}

		// calculate start and end item indexes
		const startIndex = (currentPage - 1) * pageSize
		const endIndex = Math.min(startIndex + pageSize - 1, totalItems - 1)

		const pages = [...Array(endPage + 1 - startPage).keys()].map(i => startPage + i)

		return {
			totalItems,
			currentPage,
			pageSize,
			totalPages,
			startPage,
			endPage,
			startIndex,
			endIndex,
			pages,
		}
	}

	// setPage = page => Router.push(updateQueryParam('page', page)) //@todo client side rendering
	setPage = page => {
		// for mvp only server side
		window.location.href = updateQueryParam('page', page)
	}

	render() {
		const { pager } = this.state
		const { total } = this.props

		if (!pager.pages || pager.pages.length <= 1) {
			return null
		}

		return (
			<ResultPagination>
				<Pagination>
					{pager.currentPage > 1 && (
						<>
							<li className="arrow">
								<a
									className="icon-left-double-arrow"
									title="gå til første"
									onClick={() => this.setPage(1)} // first page
								/>
							</li>
							<li className="arrow">
								<a
									className="icon-arrow-alt-left"
									title="gå til forrige"
									onClick={() => this.setPage(pager.currentPage - 1)} // previous page
								/>
							</li>
						</>
					)}

					{pager.pages.map((p, index) => (
						<li key={index} className={parseInt(pager.currentPage) === p ? 'current' : ''}>
							<a onClick={() => this.setPage(p)}>{p}</a>
						</li>
					))}

					{pager.currentPage < total && (
						<>
							<li className="arrow">
								<a
									className="icon-arrow-alt-right"
									title="gå til neste"
									onClick={() => this.setPage(pager.currentPage + 1)} // next page
								/>
							</li>
							<li className="arrow">
								<a
									className="icon-right-double-arrow"
									title="gå til siste"
									onClick={() => this.setPage(pager.totalPages)} // last page
								/>
							</li>
						</>
					)}
				</Pagination>
			</ResultPagination>
		)
	}
}

EavisSearchPagination.propTypes = propTypes
EavisSearchPagination.defaultProps = defaultProps

export default EavisSearchPagination
