import React from 'react'
import PropTypes from 'prop-types'
import { Col } from '@aller/shiny/src/atoms/Col'
import { DotLoader } from '@aller/shiny/src/atoms/loaders/DotLoader'
import { Row } from '@aller/shiny/src/atoms/Row'
import { Heading } from '@aller/shiny/src/atoms/Heading'
import { Button } from '@aller/shiny/src/atoms/Button'
import ListItem from './ListItem'
import Screen from '../Screen'
import getData from '../UserInterests/services/get-data'
import withData from '../../decorators/with-data'

class ListArticle extends React.Component {
	constructor(props) {
		super(props)
		this.state = {
			items: [],
			visible: 8,
		}
		this.loadMore = this.loadMore.bind(this)
	}

	componentWillReceiveProps(nextProps) {
		const { userData } = nextProps
		if (userData) this.setState({ items: userData.data })
	}

	loadMore() {
		this.setState(prev => {
			return { visible: prev.visible + 4 }
		})
	}

	render() {
		const { error, isLoadingData } = this.props
		const { items, visible } = this.state

		if (isLoadingData) {
			return (
				<Screen color="secondary" center>
					<DotLoader />
				</Screen>
			)
		}

		if (error) {
			return (
				<Screen color="secondary" center>
					<Heading size="huge">Oops, noe gikk galt.</Heading>
				</Screen>
			)
		}

		if (!items || items.length === 0) {
			return (
				<Screen color="secondary" center>
					<Heading size="huge">Du har ikke lest noen artikler ennå</Heading>
				</Screen>
			)
		}

		const itemsByTime = (items || []).sort((a, b) => b.timestamp - a.timestamp)

		return (
			<Screen color="secondary" center>
				<Heading size="huge">Articles you have read</Heading>
				<Row>
					{itemsByTime.slice(0, visible).map((item, i) => {
						return (
							<Col xs={12} md={3} key={i + item.title}>
								<ListItem {...item} key={i + item.title} />
							</Col>
						)
					})}
				</Row>
				<Row center="xs">
					{visible < itemsByTime.length && <Button onClick={this.loadMore}>Vis flere</Button>}
				</Row>
			</Screen>
		)
	}
}

ListArticle.propTypes = {
	isLoadingData: PropTypes.bool.isRequired,
	error: PropTypes.bool.isRequired,
	userData: PropTypes.arrayOf(PropTypes.shape({ title: PropTypes.string })).isRequired,
}

const DailyCatalog = withData({
	fetchData: getData,
	dataProp: 'userData',
	loadingProp: 'isLoadingData',
})(ListArticle)

const WeeklyCatalog = withData({
	fetchData: getData,
	dataProp: 'userData',
	loadingProp: 'isLoadingData',
})(ListArticle)

const Switcher = props => {
	const { _dataOptions } = props
	return _dataOptions.statisticType === 'daily' ? (
		<DailyCatalog {...props} />
	) : (
		<WeeklyCatalog {...props} />
	)
}
Switcher.propTypes = {
	_dataOptions: PropTypes.shape({ statisticType: PropTypes.string }).isRequired,
}

export default Switcher
