import React from 'react'
import propTypes from 'prop-types'
import dynamic from 'next/dynamic'
import { withTheme } from 'styled-components'
import { PlugHeading as DefaultHeading } from '@aller/shiny/src/atoms/PlugHeading'
import { Labels as DefaultLabels } from '@aller/shiny/src/molecules/Labels'
import { OverlayingPlugLabels } from '@aller/shiny/src/molecules/OverlayingPlugLabels'
import { themePropTypes } from '@aller/shiny/src/themes/theme-prop-types'
import OverImageHeading from './OverImageHeading'
import { fillEmptyGridSizes } from '../SkiStarPlug/get-grid-sizes'

const TrysilPlug = dynamic(() =>
	import('@aller/shiny/src/molecules/TrysilPlug').then(c => c.TrysilPlug),
)
const SkiStarPlug = dynamic(() => import('../SkiStarPlug'))
const SmallPreview = dynamic(() => import('../SkiStarPlug/SmallPreview'))

const ArticlePreview = ({
	deck,
	heading,
	teaserTitle,
	headingComponent,
	headingProps: { skin, ALLCAPS, size: propSize, maxLines },
	imageUrl,
	metadata: {
		showHeroPreview,
		isMeninger,
		hideImage,
		isRecommended,
		boxBackground: { desktop },
	},
	labelsComponent,
	standFirst,
	width,
	label,
	image,
	isNested,
	column,
	theme,
	topnormal,
	device,
	...rest
}) => {
	const autoSize = !isNested && width > 60 ? 'huge' : 'large'
	const size = propSize ? propSize : autoSize
	const UnsizedHeading = headingComponent === 'dymo' ? OverImageHeading : DefaultHeading
	const Heading = props => <UnsizedHeading size={size} {...props} />
	const CustomLabels = labelsComponent === 'overlaying' ? OverlayingPlugLabels : DefaultLabels

	const Labels = props => {
		const order = labelsComponent === 'overlaying' ? 2 : null
		return <CustomLabels {...props} order={order} />
	}

	const placeholderUrl =
		'data:image/png;base64,iVBORw0KGgoAAAANSUhEUgAAABAAAAAJCAYAAAA7KqwyAAAAF0lEQVR42mN8++b9fwYKAOOoAaMGAAEA4ukiCUCbW0cAAAAASUVORK5CYII=' // eslint-disable-line max-len
	const stripHeadingHTML = theme.name !== 'SOL'
	let title = teaserTitle || heading
	if (!stripHeadingHTML) {
		title = heading
	}

	if (isMeninger) {
		label = 'meninger'
	}

	if (isRecommended) {
		label = 'Redaksjonen anbefaler'
	}

	const props = {
		column,
		preventBlur: true,
		kicker: deck,
		title,
		rawTitle: heading,
		...{ Heading, headingProps: { skin: massageSkin(skin), ALLCAPS, maxLines } }, // eslint-disable-line no-use-before-define
		Labels,
		labradorLabel: label,
		showHeroPreview,
		isMeninger,
		isRecommended,
		isBlack: desktop === 'black',
		stripHeadingHTML,
		hideImage,
		subtitle: standFirst,
		maxsize: topnormal,
		image: placeholderUrl,
		placeholderUrl,
		sources: image.sources,
		ratio: Math.floor(image.aspectRatio * 10000) / 10000,
		offset: 400,
		...rest,
	}

	if (theme.name === 'SOL') {
		column = fillEmptyGridSizes(column)
		if (device.mobile && (column.md < 7 || (topnormal && column.md < 7))) {
			return <SmallPreview topnormal={topnormal} {...props} />
		}
		return <SkiStarPlug topnormal={topnormal} {...props} />
	}

	return <TrysilPlug {...props} />
}
ArticlePreview.propTypes = {
	deck: propTypes.string,
	heading: propTypes.string,
	teaserTitle: propTypes.string,
	headingComponent: propTypes.string,
	labelsComponent: propTypes.oneOf(['overlaying', '']),
	imageUrl: propTypes.string,
	standFirst: propTypes.string,
	label: propTypes.string,
	metadata: propTypes.shape({
		boxBackground: propTypes.shape({
			desktop: propTypes.string,
		}),
		hideImage: propTypes.bool,
		isMeninger: propTypes.bool,
		isRecommended: propTypes.bool,
		showHeroPreview: propTypes.bool,
	}),
	width: propTypes.number,
	topnormal: propTypes.bool,
	device: propTypes.shape({
		mobile: propTypes.bool,
		tablet: propTypes.bool,
		desktop: propTypes.bool,
	}),
	column: propTypes.shape({
		width: propTypes.number,
		reverse: propTypes.bool,
		xs: propTypes.number,
		sm: propTypes.number,
		md: propTypes.number,
		lg: propTypes.number,
		xsOffset: propTypes.number,
		smOffset: propTypes.number,
		mdOffset: propTypes.number,
		lgOffset: propTypes.number,
		children: propTypes.node,
	}),
	image: propTypes.shape({
		url: propTypes.string,
		placeholder: propTypes.string,
		sources: propTypes.any,
		aspectRatio: propTypes.any,
	}),
	headingProps: propTypes.shape({
		maxLines: propTypes.number,
		size: propTypes.oneOf(['small', 'medium', 'large', 'xlarge', 'huge']),
		skin: propTypes.shape({
			backgroundColor: propTypes.string,
			hoverBackgroundShade: propTypes.string,
			textColor: propTypes.string,
			needsPadding: propTypes.bool,
			textHoverShade: propTypes.string,
		}),
		ALLCAPS: propTypes.bool,
	}),
	isNested: propTypes.bool,
	theme: propTypes.shape(themePropTypes),
}
ArticlePreview.defaultProps = {
	deck: null,
	heading: null,
	imageUrl: null,
	image: {},
	teaserTitle: null,
	topnormal: false,
	device: {
		mobile: false,
		tablet: false,
		desktop: false,
	},
	width: null,
	headingComponent: 'dymo',
	label: '',
	metadata: {
		isMeninger: false,
		isRecommended: false,
		showHeroPreview: false,
		boxBackground: {},
	},
	labelsComponent: '',
	standFirst: null,
	column: null,
	headingProps: {
		size: 'large',
		skin: {
			backgroundColor: 'transparent',
			hoverBackgroundShade: 'dark',
			textColor: 'type',
			textHoverShade: 'dark',
			needsPadding: false,
		},
		ALLCAPS: false,
	},
	isNested: false,
	theme: {},
}

const massageSkin = skin => {
	const defaultSkin = ArticlePreview.defaultProps.headingProps.skin

	const backgroundColor = skin.backgroundColor ? skin.backgroundColor : defaultSkin.backgroundColor
	const hoverBackgroundShade = 'dark'
	const fallbackTextColor =
		backgroundColor === 'transparent' || backgroundColor === 'yellow'
			? defaultSkin.textColor
			: 'white'
	const textColor = skin.textColor ? skin.textColor : fallbackTextColor
	const textHoverShade = skin.textHoverShade ? skin.textHoverShade : 'dark'
	const needsPadding = backgroundColor !== 'transparent'

	return {
		backgroundColor,
		hoverBackgroundShade,
		textColor,
		textHoverShade,
		needsPadding,
	}
}

export default withTheme(ArticlePreview)
