import React, { PureComponent } from 'react'
import propTypes from 'prop-types'
import { Bling as GPT } from 'react-gpt'
import { InView } from 'react-intersection-observer'
import styled from 'styled-components'
import debug from 'debug'

import { Col } from '@aller/shiny/src/atoms/Col'
import { GenericAd } from '@aller/shiny/src/molecules/GenericAd'
import { StickyAd } from '@aller/shiny/src/molecules/StickyAd'
import { WallpaperAd } from '@aller/shiny/src/molecules/WallpaperAd'
import { FullscreenAd } from '@aller/shiny/src/molecules/FullscreenAd'

import { AdsContext } from '../../store/ads'
import InnerAd from './InnerAd'

const log = debug('ads')
const reggy = /[^/]+\/[^/]+\/(?:[0-9]{8,})$/

/* eslint-disable no-underscore-dangle */
if (typeof window !== 'undefined') {
	log('googletag')
	window.googletag = window.googletag || {}
	window.googletag.cmd = window.googletag.cmd || []

	window.googletag.cmd.push(() => {
		if (typeof window.__cmp !== 'undefined') {
			window.__cmp('getVendorConsents', null, result => {
				if (result && result.purposeConsents) {
					if (result.purposeConsents['2'] === false) {
						log('Disabled personalized ads')
						window.googletag.pubads().setRequestNonPersonalizedAds(1)
					} else {
						log('Enabled personalized ads')
						window.googletag.pubads().setRequestNonPersonalizedAds(0)
					}
				}
			})
		}
	})

	if (window.location.pathname.match(reggy)) {
		import('intersection-observer')
	}
}

const StyledObserver = styled(InView)`
	margin: 0 auto;
`

class Ad extends PureComponent {
	isArticle = false

	constructor(props) {
		super(props)

		const {
			size: { width: wrapperWidth },
		} = props

		this.state = {
			wrapperWidth,
			hideAttribution: true,
			showSettingsCog: false,
			isEmpty: false,
			isIE11: false,
			slot: null,
			articleAdLoaded: false,
		}
	}

	componentDidMount() {
		const { articleAdLoaded } = this.state

		if (window.location.pathname.match(reggy)) {
			this.isArticle = true
		}
		// Check if browser is IE, because we remove parallax on fullscreen on IE
		if (
			navigator.userAgent.indexOf('MSIE') !== -1 ||
			navigator.appVersion.indexOf('Trident/') > 0
		) {
			this.setState({
				isIE11: true,
			})
		}
		/*
			DANGEROUSLY RENDERED ARTICLES (Raw articles from Labrador wrapped in a Wolverine frame)
			With the dangerously rendered articles solution we use wolverine ads in the top, but old school am-assets ads in the article itself
			There is an issue where if wolverine ads load before am-assets, the wolverine ads will not render anything.
			So a hacky solution is to check that am-assets exists in the window every 50 ms and then call refresh if it does.
		*/
		if (this.isArticle) {
			if (!articleAdLoaded) {
				this.checkForAmAssets()
			}
		}
	}

	/**
	 * Function that depends on an external script.
	 *
	 * Let's delay the execution of this function until
	 * a magic variable has been declared on the window object
	 * by an external script.
	 *
	 * If the magic variable hasn't been set, we will
	 * wait exponentially longer for each attempt,
	 * thus reducing the resources consumed compared to
	 * trying with a fixed interval.
	 */

	checkForAmAssets = () => {
		const { slot } = this.state
		if (!window.am) {
			setTimeout(this.checkForAmAssets, (this.timeout = 2 * (this.timeout || 1)))
			log(`magicVariable is not ready, waiting ${this.timeout} milliseconds`)
			return
		}

		GPT.refresh([slot])
	}

	handeOnSlotLoad = slot => {
		this.setState({
			slot,
			articleAdLoaded: true,
		})
	}

	handleOnSlotRenderEnded = (slot, id, context) => {
		if (slot.size) {
			if (slot.size[0] === 0 || slot.size[0] === 1) {
				this.setState({
					wrapperWidth: 'auto',
				})
			} else {
				this.setState({
					wrapperWidth: `${slot.size[0]}px`,
				})
			}
		}

		if (!slot.isEmpty) {
			this.setState({
				showSettingsCog: true,
				hideAttribution: false,
			})
		}

		this.setState({
			isEmpty: slot.isEmpty,
		})

		if (id.startsWith('ad-topbanner')) {
			if (slot.size && (slot.size[0] === 980 || slot.size[0] === 970) && !this.isArticle) {
				context.updateShowSkyscraper()
			}
		}
	}

	handleOnChange = inView => {
		const { articleAdLoaded } = this.state

		if (this.isArticle && !articleAdLoaded && inView) {
			GPT.refresh()
			this.setState({
				articleAdLoaded: true,
			})
		}
	}

	render() {
		const {
			id,
			width,
			size: { height },
			theme: {
				variables: { verticalBase },
			},
		} = this.props

		const { isEmpty, isIE11, wrapperWidth, hideAttribution, showSettingsCog } = this.state

		const innerAd = (
			<InnerAd
				handleOnSlotRenderEnded={this.handleOnSlotRenderEnded}
				{...this.props}
				showSettingsCog={showSettingsCog}
			/>
		)

		const sharedProps = {
			width: wrapperWidth,
			height,
			shouldHideAttribution: hideAttribution,
			itemScope: true,
			itemType: 'http://schema.org/WPAdBlock',
			style: {
				flex: 1,
			},
		}

		if (id.startsWith('ad-skyscraper-right')) {
			return (
				<AdsContext.Consumer>
					{({ showSkyscraper }) => {
						return showSkyscraper ? (
							<StickyAd {...sharedProps} sticky="right">
								{innerAd}
							</StickyAd>
						) : null
					}}
				</AdsContext.Consumer>
			)
		}
		if (id.startsWith('ad-sticky-right')) {
			return (
				<StickyAd {...sharedProps} sticky="right">
					{innerAd}
				</StickyAd>
			)
		}
		if (id.startsWith('ad-sticky-left')) {
			return (
				<Col xs={12} md={Math.ceil((12 / 100) * width)}>
					<StickyAd {...sharedProps} sticky="left" top={verticalBase}>
						{innerAd}
					</StickyAd>
				</Col>
			)
		}
		if (id.startsWith('ad-netboard')) {
			return (
				<Col xs={12} md={Math.ceil((12 / 100) * width)}>
					<GenericAd {...sharedProps}>{innerAd}</GenericAd>
				</Col>
			)
		}
		if (id.startsWith('ad-topbanner')) {
			const wp = (
				<WallpaperAd shouldHideAttribution={false} {...sharedProps}>
					<InnerAd
						style={{
							marginLeft: -15,
							marginRight: -15,
						}}
						showSettingsCog={showSettingsCog}
						handleOnSlotRenderEnded={this.handleOnSlotRenderEnded}
						handleOnSlotLoad={this.handeOnSlotLoad}
						lazyload={false}
						{...this.props}
					/>
				</WallpaperAd>
			)

			if (this.isArticle) {
				return <StyledObserver onChange={this.handleOnChange}>{wp}</StyledObserver>
			}

			return wp
		}

		if (id.startsWith('ad-fullscreen')) {
			return (
				<FullscreenAd {...sharedProps} shouldHide={isEmpty} isIE11={isIE11}>
					{innerAd}
				</FullscreenAd>
			)
		}

		if (id.startsWith('ad-medium-rectangle')) {
			sharedProps.width = '320px'
			sharedProps.height = '250px'
		}

		return <GenericAd {...sharedProps}>{innerAd}</GenericAd>
	}
}
Ad.propTypes = {
	id: propTypes.string,
	size: propTypes.shape({
		width: propTypes.oneOfType([propTypes.string, propTypes.number]),
		height: propTypes.oneOfType([propTypes.string, propTypes.number]),
	}),
	theme: propTypes.shape({
		variables: propTypes.shape({
			verticalBase: propTypes.string.isRequired,
		}).isRequired,
	}),
	width: propTypes.number,
}
Ad.defaultProps = {
	id: null,
	size: {},
	width: 100,
	theme: {
		variables: {
			verticalBase: '2.4rem',
		},
	},
}

export default Ad
