import React from 'react'
import debug from 'debug'
import { connectToAdServer } from 'react-prebid'
import { InView } from 'react-intersection-observer'
import PropTypes from 'prop-types'
import { GenericAd } from '@aller/shiny/src/molecules/GenericAd'
import { AdSettingsCog } from '@aller/shiny/src/atoms/AdSettingsCog'
import { AdsContext } from '../../store/ads'

const log = debug('ads:prebid')
const warning = debug('ads:prebid:warning')
const reggy = /[^/]+\/[^/]+\/(?:[0-9]{8,})$/

class AdSlot extends React.Component {
	isArticle = false

	constructor(props) {
		super(props)

		this.state = {
			showSettingsCog: false,
			showSticky: false,
		}
	}

	componentDidMount() {
		if (window.location.pathname.match(reggy)) {
			this.isArticle = true
		}

		this.pushAdSlot()
	}

	pushAdSlot = () => {
		if (window.googletag && window.googletag.cmd) {
			window.googletag.cmd.push(this.handleOnSlotRenderEnded.bind(this))
		} else {
			warning('googletag is not defined, what happends next can go very wrong')
		}
	}

	onChange = inView => {
		const {
			adSlotProps: { id },
			activate,
		} = this.props

		if (inView) {
			log('activating ad: ', id)
			activate(id)
		}
	}

	handleOnSlotRenderEnded() {
		const {
			adSlotProps: { id },
			adListener,
			type,
		} = this.props

		window.googletag.pubads().addEventListener('slotRenderEnded', event => {
			if (event.slot.getSlotElementId() === id) {
				adListener(event)
			}

			if (!event.isEmpty) {
				this.setState({
					showSettingsCog: true,
				})
			}

			if (event.slot.getSlotElementId().startsWith('ad-topbanner') && type) {
				if (
					event.size &&
					(event.size[0] === 1 || event.size[0] === 980 || event.size[0] === 970) &&
					!this.isArticle
				) {
					this.setState({
						showSticky: true,
					})
				}
			}
		})
	}

	render() {
		const { wrapper, wrapperProps, rootMargin, triggerOnce, adSlotProps, type } = this.props
		const { showSettingsCog, showSticky } = this.state
		const Wrapper = wrapper
		if (type === 'sticky' && !showSticky) {
			return null
		}

		return (
			<AdsContext.Consumer>
				{context => (
					<InView rootMargin={rootMargin} triggerOnce={triggerOnce} onChange={this.onChange}>
						{({ inView, ref }) => (
							<Wrapper {...wrapperProps} ref={adSlotProps.id.includes('fullscreen') ? ref : null}>
								<div>
									{showSettingsCog && <AdSettingsCog onClick={context.openModal} />}
									<div
										className="adunit wolverine-ads"
										{...adSlotProps}
										ref={adSlotProps.id.includes('fullscreen') ? null : ref}
									/>
								</div>
							</Wrapper>
						)}
					</InView>
				)}
			</AdsContext.Consumer>
		)
	}
}

AdSlot.propTypes = {
	activate: PropTypes.func.isRequired,
	adListener: PropTypes.func,
	wrapper: PropTypes.oneOfType([PropTypes.element, PropTypes.func, PropTypes.node, PropTypes.any]),
	wrapperProps: PropTypes.shape({
		width: PropTypes.string,
		height: PropTypes.string,
		shouldHideAttribution: PropTypes.bool,
	}),
	adSlotProps: PropTypes.shape({
		id: PropTypes.string.isRequired,
		style: PropTypes.object,
	}),
	rootMargin: PropTypes.string,
	triggerOnce: PropTypes.bool,
	type: PropTypes.string,
}

AdSlot.defaultProps = {
	wrapper: GenericAd,
	adListener: () => {},
	wrapperProps: {
		width: null,
		height: null,
	},
	adSlotProps: {
		style: {},
	},
	rootMargin: '400px',
	triggerOnce: true,
	type: null,
}

export default connectToAdServer(AdSlot)
