/* eslint-disable class-methods-use-this, import/prefer-default-export */
import { DestEventTypes } from '../../eventTypes'
import { BaseTracker } from '../base'
import getVideoInfo from '../../../../services/get-video-info'

const queue = []
const IGNORED_QUEUE_TYPES = [DestEventTypes.TIME]

class LinkPulseTracker extends BaseTracker {
	constructor(lpClient, logger) {
		super(logger)
		this.lpClient = lpClient

		this.lastTS = 0

		this.options = {
			nextCategory: 'next',
		}

		this.useQueue = false
	}

	// in order to get tags
	// we need to wait until ajax request is done
	// so, storing events in queue until that
	pipeToQueue(useQueue) {
		this.useQueue = useQueue

		if (!useQueue) {
			this.processQueue()
			this.queue = []
		}
	}

	processQueue() {
		queue.forEach(e => {
			const isAllowed = !IGNORED_QUEUE_TYPES.some(t => e.type === t)

			if (isAllowed) {
				try {
					this.log(e)
				} catch (err) {
					this.logger.log(`event ${e.type} got error while processing in queue: ${err}`)
				}
			}
		})
	}

	log(e) {
		if (this.useQueue) {
			this.logger.log(`pushing to queue ${e.type}`)
			queue.push(e)
			return
		}

		const { offset, position } = e.data

		this.logger.log(`event for linkpulse ${e.type}`)
		switch (e.type) {
			case DestEventTypes.NEW_PLAYLIST_ITEM:
				this.lpClient.createLogger()
				break
			case DestEventTypes.INSCREEN:
				if (!this.lpClient.loggerInitialized()) {
					this.lpClient.createLogger()
				}

				if (!this.lpClient.getLoggedLoadstart()) {
					this.setupClient(e.data)
				}
				break
			case DestEventTypes.PLAY:
			case DestEventTypes.AUTOPLAY:
			case DestEventTypes.REPLAY:
				if (e.data.videoId === this.video) {
					this.processPlay()
				} else {
					this.processNextPlay(e)
				}

				break
			case DestEventTypes.CONTINUE:
				this.lpClient.setPaused(false)
				this.lpClient.logPlaystart(position)

				break
			case DestEventTypes.PAUSE:
				this.lpClient.setPaused(true)
				break
			case DestEventTypes.SEEK:
				this.lpClient.logSeek(offset)
				break
			case DestEventTypes.STOP:
				this.lpClient.logEnded(position)
				break
			case DestEventTypes.TIME:
				this.lpClient.logTimeupdate(position)
				break
			default:
				// do nothing
				break
		}
	}

	processNextPlay(e) {
		this.setupClient(e.data, true).then(() => {
			this.lpClient.setPaused(false)

			this.lpClient.logPlaystart(0)
		})
	}

	processPlay() {
		this.lpClient.setPaused(false)

		this.lpClient.logPlaystart(0)
	}

	setupClient({ image, title, videoType, duration, videoId }, isNext = false) {
		this.pipeToQueue(true)
		return getVideoInfo(videoId, window.top.location.origin).then(v => {
			const { tags } = v

			const videoUrl = window.top.location.href
			// This url uniquely identifies video.
			const videoArticleUrl = `${window.top.location.origin}/video/track/${videoId}`

			this.lpClient.setVideoUrl(videoArticleUrl)
			this.lpClient.setPageUrl(videoUrl)
			this.lpClient.setVideoImg(image)
			this.lpClient.setVideoTitle(title)
			if (isNext) {
				this.lpClient.setVideoCategory(this.options.nextCategory)
			}

			this.lpClient.setDuration(duration)
			this.lpClient.setPageType(`video_page.${videoType}`)

			this.lpClient.setVideoTags(tags)

			this.lpClient.logLoadstart()

			this.video = videoId

			this.pipeToQueue(false)
		})
	}
}

export { LinkPulseTracker }
