/* eslint-disable import/prefer-default-export */
import { JWEventTypes, DestEventTypes } from '../../eventTypes'
import { VideoEvent } from '../../videoEvent'
import { VideoLogger } from '../../videoLogger'

/* eslint-disable class-methods-use-this */
/**
 * Class representing a mapping source of JW player
 */
class JWEventSource {
	/**
	 * @param {*} player - JW Player instance
	 */
	constructor(player, options) {
		this.player = player

		this.logger = new VideoLogger('JW_SOURCE')

		this.watchedVideos = []

		this.options = options
	}

	/**
	 * Transforms event from source player for tracker consumption
	 * @param {*} e - raw event from player, that needs to be mapped for consumers
	 * @returns {VideoEvent} mapped destination event ready for trackers to consume
	 */
	pipe(e, cb) {
		let mappedEventType = null

		this.logger.log(`dispatching raw event ${JSON.stringify(e)}`)

		const eventData = this.getEventData(e)

		switch (e.type) {
			case JWEventTypes.READY:
				mappedEventType = DestEventTypes.READY
				break
			case JWEventTypes.CONTROLS:
				mappedEventType = DestEventTypes.CONTROLS
				break
			case JWEventTypes.SEEK:
				mappedEventType = DestEventTypes.SEEK
				eventData.offset = e.offset
				break
			case JWEventTypes.TIME:
				mappedEventType = DestEventTypes.TIME
				break
			case JWEventTypes.PLAY:
				if (this.replay) {
					mappedEventType = DestEventTypes.REPLAY
					this.replay = false
					break
				}

				if (e.playReason === 'autostart' || e.playReason === 'related-auto') {
					mappedEventType = DestEventTypes.AUTOPLAY
				} else if (e.playReason === 'interaction' || e.playReason === 'related-interaction') {
					if (eventData.position && eventData.position > 0) {
						mappedEventType = DestEventTypes.CONTINUE
					} else {
						mappedEventType = DestEventTypes.PLAY
					}
				}
				break
			case JWEventTypes.PAUSE:
				mappedEventType = DestEventTypes.PAUSE
				break
			case JWEventTypes.PLAYLIST_ITEM:
				if (this.watchedVideos.includes(eventData.videoId)) {
					this.replay = true
				} else {
					this.watchedVideos.push(eventData.videoId)
				}

				mappedEventType = DestEventTypes.NEW_PLAYLIST_ITEM

				break
			case JWEventTypes.COMPLETE:
				mappedEventType = DestEventTypes.STOP
				break
			case JWEventTypes.VIEWABLE:
				if (e.viewable) {
					mappedEventType = DestEventTypes.INSCREEN
				}
				break
			default:
				cb(null) // basically, do nothing
				return
		}

		if (mappedEventType === null) {
			cb(null)
			return
		}

		const result = new VideoEvent(mappedEventType, eventData)
		cb(result)
	}

	/**
	 * Returns event for tracker consumption without type transforming
	 * @param {*} e - raw event from player, that needs to be mapped for consumers
	 * @returns {VideoEvent} mapped destination event ready for trackers to consume
	 */
	pipeRaw(e, cb) {
		const eventData = this.getEventData(e)

		this.logger.log(`dispatching raw event (pipeRaw) ${JSON.stringify(e)}`)

		if (e.type !== JWEventTypes.TIME) {
			const result = new VideoEvent(e.type, eventData)
			cb(result)
		}

		cb(null)
	}

	/**
	 * @function
	 * Return initial data for convenient event mapping
	 * @param {*} e - raw event from JW player
	 */
	getEventData(e) {
		const duration = this.player.getDuration()
		const playlistItem = this.player.getPlaylistItem()

		return {
			videoId: playlistItem.mediaid,
			title: playlistItem.title,
			duration,
			videoType: this.options.videoType,
			file: playlistItem.file,
			image: playlistItem.image,
			position: this.player.getPosition(),
		}
	}
}

export { JWEventSource }
